#pragma once

#include <yplatform/net/dns/resolver_service.h>
#include <boost/asio/basic_io_object.hpp>

namespace yplatform { namespace net { namespace dns {

// Asynchronous resolver interface class.
// Performs lookups of DNS resource records A, AAAA, PTR, MX, TXT.
//
// @par Handler is any callable object with signature:
// @code void (const error_code&, iterator) @endcode
// where iterator is one of resolver::<iterator> types depends
// in RR type: iterator_a. iterator_aaaa, iterator_ptr, iterator_mx,
// iterator_txt.
class resolver : public boost::asio::basic_io_object<resolver_service>
{
public:
    typedef resolver_service::iterator_a iterator_a;
    typedef resolver_service::iterator_aaaa iterator_aaaa;
    typedef resolver_service::iterator_ptr iterator_ptr;
    typedef resolver_service::iterator_mx iterator_mx;
    typedef resolver_service::iterator_txt iterator_txt;

    resolver(boost::asio::io_service& io) : boost::asio::basic_io_object<resolver_service>(io)
    {
    }

    resolver(boost::asio::io_service::strand& strand)
        : boost::asio::basic_io_object<resolver_service>(strand.get_io_service())
    {
        get_implementation().strand = &strand;
    }

    resolver(resolver&&) = default;
    resolver& operator=(resolver&&) = default;

    template <typename Handler>
    void async_resolve_a(const std::string& q, Handler&& handler)
    {
        get_service().async_resolve_a(get_implementation(), q, std::forward<Handler>(handler));
    }

    // Resolve IPv6 addresses
    template <typename Handler>
    void async_resolve_aaaa(const std::string& q, Handler&& handler)
    {
        get_service().async_resolve_aaaa(get_implementation(), q, std::forward<Handler>(handler));
    }

    // Back resolve
    template <typename Handler>
    void async_resolve_ptr(const std::string& q, Handler&& handler)
    {
        get_service().async_resolve_ptr(get_implementation(), q, std::forward<Handler>(handler));
    }

    // Make MX (mail exchanger) records lookup
    template <typename Handler>
    void async_resolve_mx(const std::string& q, Handler&& handler)
    {
        get_service().async_resolve_mx(get_implementation(), q, std::forward<Handler>(handler));
    }

    // Lookup TXT records
    template <typename Handler>
    void async_resolve_txt(const std::string& q, Handler&& handler)
    {
        get_service().async_resolve_mx(get_implementation(), q, std::forward<Handler>(handler));
    }

    void cancel()
    {
        get_service().cancel(get_implementation());
    }
};

}}}
