#pragma once

#include <yplatform/net/types.h>
#include <yplatform/net/ssl_settings.h>
#include <yplatform/net/dns/resolver.h>
#include <yplatform/net/dns/resolver_service.h>
#include <yplatform/net/dns/resolver_settings.h>
#include <yplatform/reactor.h>
#include <boost/asio/io_service.hpp>

namespace yplatform { namespace net {

// Data container for:
//   io_service
//   ssl context
//   dns resolver
class io_data
{
public:
    io_data() : io_(0)
    {
    }

    io_data(boost::asio::io_service& io) : io_(&io)
    {
    }

    io_data(boost::asio::io_service& io, std::unique_ptr<ssl_context_t> ssl_context)
        : io_(&io), ssl_context_(std::move(ssl_context))
    {
    }

    io_data(const io_data& o) = delete;
    io_data& operator=(const io_data&) = delete;
    io_data(io_data&&) = default;
    io_data& operator=(io_data&&) = default;

    boost::asio::io_service* get_io()
    {
        return io_;
    }

    const ssl_settings& get_ssl_settings() const
    {
        return ssl_settings_;
    }

    ssl_context_t* get_ssl_context() const
    {
        return ssl_context_.get();
    }

    void setup_ssl(const ssl_settings& settings);

    void setup_dns(
        const dns::resolver_service_settings& dns_settings = dns::resolver_service_settings())
    {
        boost::asio::use_service<dns::resolver_service>(*io_).setup_service(dns_settings);
    }

private:
    boost::asio::io_service* io_;
    ssl_settings ssl_settings_;
    std::unique_ptr<ssl_context_t> ssl_context_;
};

inline void io_data::setup_ssl(const ssl_settings& settings)
{
    ssl_settings_ = settings;
    ssl_context_.reset(new ssl_context_t(settings.method));
    ssl_context_->set_options(
        boost::asio::ssl::context::default_workarounds | boost::asio::ssl::context::no_compression |
        boost::asio::ssl::context::no_sslv2);
    if (settings.only_tls)
    {
        ssl_context_->set_options(boost::asio::ssl::context::no_sslv3);
    }
    ssl_context_->set_verify_mode(settings.verify_mode);
    if (settings.verify_file.size())
    {
        ssl_context_->load_verify_file(settings.verify_file);
    }
    ssl_context_->set_default_verify_paths();
    ssl_context_->set_options(settings.options);
    if (settings.password.size())
    {
        ssl_context_->set_password_callback(
            std::bind([password = settings.password]() -> std::string { return password; }));
    }
    if (settings.cert.size())
    {
        ssl_context_->use_certificate_chain(
            boost::asio::const_buffer(settings.cert.data(), settings.cert.size()));
    }
    else if (settings.cert_file.size())
    {
        try
        {
            ssl_context_->use_certificate_chain_file(settings.cert_file);
        }
        catch (std::exception const& e)
        {
            YLOG_LOCAL(error) << "error: can't read SSL certificate: " << settings.cert_file << ": "
                              << e.what() << "\n";
            throw;
        }
    }
    if (settings.key.size())
    {
        ssl_context_->use_private_key(
            boost::asio::const_buffer(settings.key.data(), settings.key.size()),
            boost::asio::ssl::context::pem);
    }
    else if (settings.key_file.size())
    {
        try
        {
            ssl_context_->use_private_key_file(settings.key_file, ssl_context_t::pem);
        }
        catch (std::exception const& e)
        {
            YLOG_LOCAL(error) << "error: can't read SSL private key: " << settings.key_file << ": "
                              << e.what() << "\n";
            throw;
        }
    }
    if (settings.dh_file.size())
    {
        try
        {
            ssl_context_->use_tmp_dh_file(settings.dh_file);
            ssl_context_->set_options(boost::asio::ssl::context::single_dh_use);
        }
        catch (std::exception const& e)
        {
            YLOG_LOCAL(error) << "warning: can't open " << settings.dh_file << ": " << e.what()
                              << "\n";
        }
    }
    if (settings.ciphers_list.size())
    {
        SSL_CTX_set_cipher_list(ssl_context_->native_handle(), settings.ciphers_list.c_str());
        if (settings.prefer_server_ciphers)
        {
            SSL_CTX_set_options(ssl_context_->native_handle(), SSL_OP_CIPHER_SERVER_PREFERENCE);
        }
    }
    if (settings.named_curve_id)
    {
        EC_KEY* ecdh = EC_KEY_new_by_curve_name(settings.named_curve_id);
        if (ecdh == NULL)
        {
            throw std::logic_error("unable to create ECC curve");
        }
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wold-style-cast"
        SSL_CTX_set_options(ssl_context_->native_handle(), SSL_OP_SINGLE_ECDH_USE);
        SSL_CTX_set_tmp_ecdh(ssl_context_->native_handle(), ecdh);
#pragma GCC diagnostic pop
        EC_KEY_free(ecdh);
    }
}

// backward compatibility hacks
typedef io_data base_service;

}}
