#pragma once

#include <yplatform/net/io_data.h>
#include <yplatform/net/ssl_settings.h>
#include <yplatform/reactor.h>
#include <map>

namespace yplatform { namespace net {

/**
 *   Reactor wrapper, holds the io_pools
 * with io_data instances associated with each
 * pool in this reactor.
 */
class io_service
{
public:
    io_service(const ::yplatform::reactor_ptr& reactor) : reactor_(reactor)
    {
    }

    /**
     * If called from a thread of reactor, return its io_data,
     * otherwise, return io_data linked with reactor using round-robin
     * scheme.
     */
    io_data* get_io_data()
    {
        return &data_[reactor_->get_index()];
    }

    const ::yplatform::reactor_ptr& reactor() const
    {
        return reactor_;
    }

    std::size_t get_index() const
    {
        return reactor_->get_index();
    }

    io_data& operator[](size_t i)
    {
        return data_[i];
    }

    size_t size() const
    {
        return data_.size();
    }

    void init()
    {
        data_.reserve(reactor_->size());
        for (std::size_t i = 0; i < reactor_->size(); ++i)
        {
            auto io = (*reactor_)[i]->io();
            std::unique_ptr<ssl_context_t> ssl_ctx(
                new ssl_context_t(boost::asio::ssl::context::sslv23));
            ssl_ctx->set_options(
                boost::asio::ssl::context::default_workarounds |
                boost::asio::ssl::context::no_compression | boost::asio::ssl::context::no_sslv2 |
                boost::asio::ssl::context::no_sslv3);
            ssl_ctx->set_verify_mode(boost::asio::ssl::context::verify_none);
            data_.emplace_back(*io, std::move(ssl_ctx));
        }
    }

    void create_ssl_context(const ssl_settings& settings)
    {
        for (auto& iodata : data_)
        {
            iodata.setup_ssl(settings);
        }
    }

    void setup_dns(const dns::resolver_service_settings& settings)
    {
        for (auto& io_data : data_)
        {
            io_data.setup_dns(settings);
        }
    }

protected:
    const std::vector<io_data>& get_io_data_array() const
    {
        return data_;
    }

private:
    std::vector<io_data> data_;
    ::yplatform::reactor_ptr reactor_;
};

}}
