#pragma once

#include <yplatform/net/server.h>
#include <yplatform/net/client.h>
#include <yplatform/net/find_io_pool.h>
#include <yplatform/log.h>
#include <yplatform/configuration.h>
#include <yplatform/ptree.h>

namespace yplatform { namespace net {

template <typename Session, typename Server = yplatform::net::server<Session>>
class server_module
{
    typedef typename Session::settings_t settings_t;

public:
    typedef std::deque<std::pair<string, unsigned>> endpoint_list;
    typedef Server server_t;

    server_module(const string& name) : service_name_(name)
    {
    }

    bool open_server(
        const yplatform::ptree& data,
        const yplatform::log::source& logger = YGLOBAL_LOGGER)
    {
        return open_server(data, 0, logger);
    }

    bool open_server(
        const yplatform::ptree& data,
        settings_t* st,
        const yplatform::log::source& logger = YGLOBAL_LOGGER);
    bool open_server(
        yplatform::reactor& reactor,
        const yplatform::ptree& data,
        settings_t* st,
        const yplatform::log::source& logger = YGLOBAL_LOGGER);
    void run_server()
    {
    }

    void stop_server()
    {
        server_->stop_server();
    }

    void abort_server()
    {
        server_->abort_server();
    }

    const endpoint_list& local_endpoints() const
    {
        return local_endpoints_;
    }

    boost::asio::io_service* io()
    {
        return server_->get_io();
    }

protected:
    void create_server(const yplatform::ptree& data, settings_t* st);

    boost::shared_ptr<server_t> server_;
    yplatform::net::ssl_settings ssl_settings_;
    string service_name_;
    endpoint_list local_endpoints_;
};

template <typename Session, typename Server>
bool server_module<Session, Server>::open_server(
    const yplatform::ptree& data,
    settings_t* default_settings,
    const yplatform::log::source& logger)
{
    return open_server(
        *find_or_create_reactor(data, service_name_), data, default_settings, logger);
}

template <typename Session, typename Server>
bool server_module<Session, Server>::open_server(
    yplatform::reactor& reactor,
    const yplatform::ptree& data,
    settings_t* default_settings,
    const yplatform::log::source& logger)
{
    server_.reset(new server_t(yplatform::reactor::make_not_owning_copy(reactor)));
    server_->logger(logger);
    boost::optional<const yplatform::ptree&> ssl_cont = data.get_child_optional("ssl_context");

    if (ssl_cont)
    {
        ssl_settings_.parse_ptree(ssl_cont.get());
        server_->create_ssl_context(ssl_settings_);
    }
    // dns settings
    auto dns_conf = data.get_child_optional("dns");
    if (dns_conf)
    {
        dns::resolver_service_settings dns_settings;
        dns_settings.parse(*dns_conf);
        server_->setup_dns(dns_settings);
    }
    // common setting
    boost::optional<const yplatform::ptree&> default_settings_tree =
        data.get_child_optional("default");
    std::unique_ptr<settings_t> default_settings_raii;
    if (default_settings_tree)
    {
        if (!default_settings)
        {
            default_settings_raii.reset(new settings_t());
            default_settings = default_settings_raii.get();
        }
        default_settings->parse_ptree(default_settings_tree.get());
    }
    // endpoints
    boost::optional<const yplatform::ptree&> data_tree = data.get_child_optional("endpoints");
    if (!data_tree) return false;
    for (const auto& endp : data_tree.get())
    {
        if (endp.first != "listen") continue;
        auto addr_optional = endp.second.get_optional<std::string>("<xmlattr>.addr");
        auto addr =
            addr_optional ? addr_optional.value() : endp.second.get<std::string>("addr", "");
        auto port_optional = endp.second.get_optional<unsigned short>("<xmlattr>.port");
        auto port =
            port_optional ? port_optional.value() : endp.second.get<unsigned short>("port", 0);
        typename Session::settings_t settings;
        if (default_settings) settings = *default_settings;
        settings.parse_ptree(endp.second);
        if (!server_->listen(settings, addr, port)) return false;
        local_endpoints_.push_back(std::make_pair(addr, port));
    }
    return true;
}

template <typename Session, typename Client = yplatform::net::client<Session>>
class client_module
{
    typedef typename Session::settings_t settings_t;
    typedef Client client_t;

public:
    client_module(const string& name) : service_name_(name)
    {
    }

    bool open_client(const yplatform::ptree& data, const log::source& logger = YGLOBAL_LOGGER)
    {
        return open_client(data, 0, logger);
    }
    bool open_client(
        const yplatform::ptree& data,
        settings_t* default_settings,
        const log::source& logger = YGLOBAL_LOGGER);
    bool open_client(
        yplatform::reactor& reactor,
        const yplatform::ptree& data,
        settings_t* default_settings,
        const log::source& logger = YGLOBAL_LOGGER);
    void run_client()
    {
    }

    void stop_client()
    {
    }

    boost::shared_ptr<Session> create_session()
    {
        return client_->create_session(settings_);
    }

    template <typename Allocator>
    boost::shared_ptr<Session> create_session(const Allocator& allocator)
    {
        return client_->create_session(settings_, allocator);
    }

    boost::asio::io_service* io()
    {
        return client_->get_io();
    }

protected:
    boost::shared_ptr<client_t> client_;
    settings_t settings_;
    yplatform::net::ssl_settings ssl_settings_;
    string service_name_;
};

template <typename Session, typename Client>
bool client_module<Session, Client>::open_client(
    const yplatform::ptree& data,
    settings_t* default_settings,
    const yplatform::log::source& logger)
{
    return open_client(
        *find_or_create_reactor(data, service_name_), data, default_settings, logger);
}

template <typename Session, typename Client>
bool client_module<Session, Client>::open_client(
    yplatform::reactor& reactor,
    const yplatform::ptree& data,
    settings_t* default_settings,
    const yplatform::log::source& logger)
{
    client_.reset(new client_t(yplatform::reactor::make_not_owning_copy(reactor)));
    client_->logger(logger);
    // common setting
    boost::optional<const yplatform::ptree&> default_settings_tree =
        data.get_child_optional("default");
    std::unique_ptr<settings_t> default_settings_raii;
    if (default_settings_tree)
    {
        if (!default_settings)
        {
            default_settings_raii.reset(new settings_t());
            default_settings = default_settings_raii.get();
        }
        default_settings->parse_ptree(default_settings_tree.get());
    }

    // ssl settings
    boost::optional<const yplatform::ptree&> ssl_cont = data.get_child_optional("ssl_context");
    if (ssl_cont)
    {
        ssl_settings_.parse_ptree(ssl_cont.get());
        client_->create_ssl_context(ssl_settings_);
    }

    // dns settings
    auto dns_conf = data.get_child_optional("dns");
    if (dns_conf)
    {
        dns::resolver_service_settings dns_settings;
        dns_settings.parse(*dns_conf);
        client_->setup_dns(dns_settings);
    }
    // net_settings
    boost::optional<const yplatform::ptree&> data_tree = data.get_child_optional("net_settings");
    if (!data_tree && !default_settings) return false;
    if (default_settings) settings_ = *default_settings;
    if (data_tree) settings_.parse_ptree(data_tree.get());
    return true;
}

}}
