#pragma once

#include <boost/noncopyable.hpp>
#include <boost/thread.hpp>
#include <boost/asio.hpp>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/system/error_code.hpp>
#include <boost/system/system_error.hpp>
#include <boost/exception_ptr.hpp>
#include <boost/enable_shared_from_this.hpp>

// The socket methods are MT unsafe.
// Here is adaptor to use these methods in a safe way.

namespace yplatform { namespace net {

// Implementation - raw socket type

template <
    typename Implementation,
    typename Mutex = boost::mutex,
    typename Lock = boost::lock_guard<Mutex>>
class protected_socket_type
    : public boost::asio::socket_base
    , private boost::noncopyable
{
    typedef boost::system::error_code error_code_type;
    typedef boost::system::system_error system_error;

public:
    typedef Implementation implementation_type;

    typedef Mutex mutex_type;
    typedef Lock lock_type;

    typedef typename boost::asio::socket_base::shutdown_type shutdown_type;
    typedef typename implementation_type::endpoint_type endpoint_type;
    typedef typename implementation_type::lowest_layer_type lowest_layer_type;

    protected_socket_type(implementation_type& raw, mutex_type& mux) : raw_(raw), mutex_(mux)
    {
    }

    lowest_layer_type& lowest_layer()
    {
        return raw_.lowest_layer();
    }
    lowest_layer_type const& lowest_layer() const
    {
        return raw_.lowest_layer();
    }

    implementation_type& raw()
    {
        return raw_;
    }
    implementation_type const& raw() const
    {
        return raw_;
    }

    boost::asio::io_service& get_io_service()
    {
        // Consider it MT safe
        return raw().get_io_service();
    }

    error_code_type cancel(error_code_type& ec)
    {
        lock_type lock(mutex());
        return raw().cancel(ec);
    }

    void cancel()
    {
        lock_type lock(mutex());
        raw().cancel();
    }

    error_code_type close(error_code_type& ec)
    {
        lock_type lock(mutex());
        return raw().close(ec);
    }

    void close()
    {
        lock_type lock(mutex());
        return raw().close();
    }

    bool is_open() const
    {
        lock_type lock(mutex());
        return raw().is_open();
    }

    std::size_t available(error_code_type& ec) const
    {
        lock_type lock(mutex());
        return raw().available(ec);
    }

    std::size_t available() const
    {
        lock_type lock(mutex());
        return raw().available();
    }

    endpoint_type remote_endpoint(error_code_type& ec) const
    {
        lock_type lock(mutex());
        return raw().remote_endpoint(ec);
    }

    endpoint_type remote_endpoint() const
    {
        lock_type lock(mutex());
        return raw().remote_endpoint();
    }

    error_code_type shutdown(shutdown_type what, error_code_type& ec)
    {
        lock_type lock(mutex());
        return raw().shutdown(what, ec);
    }

    void shutdown(shutdown_type what)
    {
        lock_type lock(mutex());
        raw().shutdown(what);
    }

    template <typename ConstBufferSequence>
    std::size_t write_some(const ConstBufferSequence& buffers, error_code_type& ec)
    {
        lock_type lock(mutex());
        return raw().write_some(buffers, ec);
    }

    template <typename ConstBufferSequence>
    std::size_t write_some(const ConstBufferSequence& buffers)
    {
        lock_type lock(mutex());
        return raw().write_some(buffers);
    }

    template <typename MutableBufferSequence>
    std::size_t read_some(const MutableBufferSequence& buffers, error_code_type& ec)
    {
        lock_type lock(mutex());
        return raw().read_some(buffers, ec);
    }

    template <typename MutableBufferSequence>
    std::size_t read_some(const MutableBufferSequence& buffers)
    {
        lock_type lock(mutex());
        return raw().read_some(buffers);
    }

    template <typename ConstBufferSequence, typename WriteHandler>
    void async_write_some(const ConstBufferSequence& buffers, WriteHandler handler)
    {
        lock_type lock(mutex());
        raw().async_write_some(buffers, handler);
    }

    template <typename MutableBufferSequence, typename ReadHandler>
    void async_read_some(const MutableBufferSequence& buffers, ReadHandler handler)
    {
        lock_type lock(mutex());
        raw().async_read_some(buffers, handler);
    }

protected:
    inline mutex_type& mutex() const
    {
        return mutex_;
    }

private:
    implementation_type& raw_;
    mutex_type& mutex_;
};

}} // namespace yplatform::net
