#pragma once

#include <boost/thread.hpp>
#include <boost/asio.hpp>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/system/error_code.hpp>
#include <boost/system/system_error.hpp>
#include <boost/exception_ptr.hpp>
#include <boost/enable_shared_from_this.hpp>

// The socket methods are MT unsafe. Here is adaptor to use the method in safe
// way.

namespace yplatform { namespace net {

// Implementation - raw socket type
// Owner - is typically socket owner (shared ptr to the session or something
// like that)

template <typename Strand, typename Implementation>
class protected_socket_type : public boost::asio::socket_base
{
    typedef boost::system::error_code error_code_type;
    typedef boost::system::system_error system_error;

public:
    typedef Strand strand_type;
    typedef Implementation implementation_type;

    typedef typename boost::asio::socket_base::shutdown_type shutdown_type;
    typedef typename implementation_type::endpoint_type endpoint_type;

    protected_socket_type(strand_type& strand, implementation_type& raw)
        : strand_(&strand), raw_(&raw)
    {
    }

    implementation_type& raw()
    {
        return *raw_;
    }
    implementation_type const& raw() const
    {
        return *raw_;
    }

    boost::asio::io_service& get_io_service()
    {
        // Consider MT safe
        return raw().get_io_service();
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    error_code_type cancel(error_code_type& ec)
    {
        sync_call(boost::bind(&implementation_type::cancel, _1, boost::ref(ec)));
        return ec;
    }

    void cancel()
    {
        boost::system::error_code ec;
        cancel(ec);
        if (ec) throw system_error(ec);
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    boost::system::error_code& close(boost::system::error_code& ec)
    {
        sync_call(boost::bind(&implementation_type::close, _1, boost::ref(ec)));
        return ec;
    }

    void close()
    {
        error_code_type ec;
        if (close(ec)) throw system_error(ec);
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    bool is_open() const
    {
#if 1 // considering MT safe
        return raw_->> is_open();
#else
        return sync_call_with_return<bool>(boost::bind(&implementation_type::is_open, _1));
#endif
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    std::size_t available(error_code_type& ec) const
    {
        return sync_call_with_return<std::size_t>(
            boost::bind(&implementation_type::available, _1, boost::ref(ec)));
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    std::size_t available() const
    {
        error_code_type ec;
        std::size_t ret = available(ec);
        if (ec) throw system_error(ec);
        return ret;
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    endpoint_type remote_endpoint(error_code_type& ec) const
    {
#if 1 // consider MT safe
        return raw_->remote_endpoint(ec);
#else
        return sync_call_with_return<endpoint_type>(
            boost::bind(&implementation_type::remote_endpoint, _1, boost::ref(ec)));
#endif
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    endpoint_type remote_endpoint() const
    {
        error_code_type ec;
        std::size_t ret = remote_endpoint(ec);
        if (ec) throw system_error(ec);
        return ret;
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    boost::system::error_code shutdown(shutdown_type what, error_code_type& ec)
    {
#if 1 // consider MT safe
        return raw_->shutdown(what, ec);
#else
        sync_call(boost::bind(&implementation_type::shutdown, _1, what, boost::ref(ec)));
        return ec;
#endif
    }

    void shutdown(shutdown_type what)
    {
        error_code_type ec;
        if (shutdown(what, ec)) throw system_error(ec);
    }

    template <typename ConstBufferSequence>
    std::size_t write_some(const ConstBufferSequence& buffers, error_code_type& ec)
    {
#if 1 // consider MT safe
        return raw_->write_some(buffers, ec);
#else
        return sync_call_with_return<std::size_t>(boost::bind(
            &implementation_type::template write_some<ConstBufferSequence>,
            _1,
            boost::ref(buffers),
            boost::ref(ec)));
#endif
    }

    template <typename ConstBufferSequence>
    std::size_t write_some(const ConstBufferSequence& buffers)
    {
        error_code_type ec;
        std::size_t ret = write_some(buffers, ec);
        if (ec) throw system_error(ec);
        return ret;
    }

    template <typename MutableBufferSequence>
    std::size_t read_some(const MutableBufferSequence& buffers, error_code_type& ec)
    {
#if 1 // consider MT safe
        return raw_->read_some(buffers, ec);
#else
        return sync_call_with_return<std::size_t>(boost::bind(
            &implementation_type::template read_some<MutableBufferSequence>,
            _1,
            boost::ref(buffers),
            boost::ref(ec)));
#endif
    }

    template <typename MutableBufferSequence>
    std::size_t read_some(const MutableBufferSequence& buffers)
    {
        error_code_type ec;
        std::size_t ret = read_some(buffers, ec);
        if (ec) throw system_error(ec);
        return ret;
    }

protected:
    inline strand_type& strand() const
    {
        return *strand_;
    }

    void sync_call(boost::function<void(implementation_type&)> const& func) const
    {
        boost::exception_ptr error;
        boost::mutex mux;
        boost::condition_variable cond;
        volatile bool complete = false;

        using boost::ref;
        using boost::cref;

        strand().dispatch(boost::bind(
            &protected_socket_type::sync_call_impl,
            this,
            cref(func),
            ref(complete),
            ref(error),
            ref(mux),
            ref(cond)));

        boost::unique_lock<boost::mutex> lock(mux);
        while (!complete)
            cond.wait(lock);

        if (error) boost::rethrow_exception(error);
    }

    void sync_call_impl(
        boost::function<void(implementation_type&)> const& func,
        volatile bool& complete,
        boost::exception_ptr& error,
        boost::mutex& mux,
        boost::condition_variable& cond) const
    {
        try
        {
            func(*raw_);
            error = boost::exception_ptr();
        }
        catch (...)
        {
            error = boost::current_exception();
        }

        boost::unique_lock<boost::mutex> lock(mux);
        complete = true;
        cond.notify_all();
    }

    template <typename R>
    R sync_call_with_return(boost::function<R(implementation_type&)> const& func) const
    {
        R ret = R(); /* XXX is it proper initialization?*/

        boost::exception_ptr error;
        boost::mutex mux;
        boost::condition_variable cond;
        volatile bool complete = false;

        using boost::ref;
        using boost::cref;

        strand().dispatch(boost::bind(
            &protected_socket_type::sync_call_with_return_impl<R>,
            this,
            cref(func),
            ref(ret),
            ref(complete),
            ref(error),
            ref(mux),
            ref(cond)));

        boost::unique_lock<boost::mutex> lock(mux);
        while (!complete)
            cond.wait(lock);

        if (error) boost::rethrow_exception(error);

        return ret;
    }

    template <typename R>
    void sync_call_with_return_impl(
        boost::function<R(implementation_type&)> const& func,
        R& ret,
        volatile bool& complete,
        boost::exception_ptr& error,
        boost::mutex& mux,
        boost::condition_variable& cond) const
    {
        try
        {
            ret = func(*raw_);
            error = boost::exception_ptr();
        }
        catch (...)
        {
            error = boost::current_exception();
        }
        boost::unique_lock<boost::mutex> lock(mux);
        complete = true;
        cond.notify_all();
    }

private:
    // pointers are needed to provide copy semantic
    strand_type* strand_;
    implementation_type* raw_;

protected:
    // copying is possible via protected_socket_with_owner
    protected_socket_type(protected_socket_type const& other)
        : strand_(other.strand_), raw_(other.raw_)
    {
    }

    protected_socket_type& operator=(protected_socket_type const& other)
    {
        strand_ = other.strand_;
        raw_ = other.raw_;
        return *this;
    }
};

// with owner and copy semanthic
// ensures that owner will not be destroyed before socket
template <typename Strand, typename Implementation, typename Owner>
class protected_socket_with_owner
    : public protected_socket_type<Strand, Implementation>
    , boost::enable_shared_from_this<protected_socket_with_owner<Strand, Implementation, Owner>>
{
    typedef protected_socket_type<Strand, Implementation> protected_socket_t;

public:
    typedef typename base_t::strand_type strand_type;
    typedef typename base_t::implementation_type implementation_type;

    typedef Owner owner_type;

    protected_socket_with_owner(
        strand_type& strand,
        implementation_type& raw,
        owner_type owner = owner_type())
        : protected_socket_(new protected_socket_t(strand, raw)), owner_(owner)
    {
    }

    protected_socket_with_owner(protected_socket_with_owner const& other)
        : protected_socket_(other.protected_socket_), owner_(other.owner)
    {
    }

    protected_socket_with_owner& operator=(protected_socket_with_owner const& other)
    {
        protected_socket_ = other.protected_socket_;
        owner_ = other.owner;
    }

    // protected socket functions
    boost::asio::io_service& get_io_service()
    {
        return protected_socket_->.get_io_service();
    }

    error_code_type cancel(error_code_type& ec)
    {
        return protected_socket_->cancel(ec);
    }

    void cancel()
    {
        protected_socket_->cancel();
    }

    boost::system::error_code& close(boost::system::error_code& ec)
    {
        return protected_socket_->close(ec);
    }

    void close()
    {
        protected_socket_->close();
    }

    bool is_open() const
    {
        return protected_socket_->is_open();
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    std::size_t available(error_code_type& ec) const
    {
        return protected_socket_->available(ec);
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    std::size_t available() const
    {
        return protected_socket_->available();
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    endpoint_type remote_endpoint(error_code_type& ec) const
    {
        return protected_socket_->remote_endpoint(ec);
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    endpoint_type remote_endpoint() const
    {
        return protected_socket_->remote_endpoint();
    }

    // SHOULD NOT be called under same strand or deadlock will happen.
    boost::system::error_code shutdown(shutdown_type what, error_code_type& ec)
    {
        return protected_socket_->shutdown(what, ec);
    }

    void shutdown(shutdown_type what)
    {
        return protected_socket_->shutdown(what);
    }

    template <typename ConstBufferSequence>
    std::size_t write_some(const ConstBufferSequence& buffers, error_code_type& ec)
    {
        return protected_socket_->write_some(buffers, ec);
    }

    template <typename ConstBufferSequence>
    std::size_t write_some(const ConstBufferSequence& buffers)
    {
        return protected_socket_->write_some(buffers);
    }

    template <typename MutableBufferSequence>
    std::size_t read_some(const MutableBufferSequence& buffers, error_code_type& ec)
    {
        return protected_socket_->read_some(buffers, ec);
    }

    template <typename MutableBufferSequence>
    std::size_t read_some(const MutableBufferSequence& buffers)
    {
        return protected_socket_->read_some(buffers);
    }

    // New methods
    void async_cancel()
    {
        async_call(boost::bind(&implementation_type::cancel, _1));
    }

    void async_close()
    {
        async_call(boost::bind(&implementation_type::close, _1));
    }

    template <typename ConstBufferSequence, typename WriteHandler>
    void async_write_some(const ConstBufferSequence& buffers, WriteHandler handler)
    {
        async_call(boost::bind(
            &implementation_type::template async_write_some<ConstBufferSequence, WriteHandler>,
            _1,
            boost::ref(buffers),
            handler));
    }

    template <typename MutableBufferSequence, typename ReadHandler>
    void async_read_some(const MutableBufferSequence& buffers, ReadHandler handler)
    {
        async_call(boost::bind(
            &implementation_type::template async_read_some<MutableBufferSequence, ReadHandler>,
            _1,
            boost::ref(buffers),
            handler));
    }

protected:
    void async_call(boost::function<void(implementation_type&)> const& func)
    {
        this->strand().post(boost::bind(&protected_socket_with_owner::async_call_impl, this, func));
    }

    void async_call_impl(boost::function<void(implementation_type&)> const& func)
    {
        try
        {
            func(this->raw());
        }
        catch (...)
        {
        }
    }

private:
    boost::shared_ptr<protected_socket_t> protected_socket_;
    owner_type& owner_;
};

}} // namespace yplatform::net
