#pragma once
#include <boost/asio.hpp>

// reposts handler in io_service to make it run out of strand

namespace yplatform { namespace net {

namespace detail {

template <typename H, typename A1>
struct Func1
{
    Func1(H const& h, A1 const& a1) : h(h), a1(a1)
    {
    }
    void operator()()
    {
        h(a1);
    }
    void operator()() const
    {
        h(a1);
    }
    H h;
    A1 a1;
};

template <typename H, typename A1, typename A2>
struct Func2
{
    Func2(H const& h, A1 const& a1, A2 const& a2) : h(h), a1(a1), a2(a2)
    {
    }
    void operator()()
    {
        h(a1, a2);
    }
    void operator()() const
    {
        h(a1, a2);
    }
    H h;
    A1 a1;
    A2 a2;
};

template <typename H, typename A1, typename A2, typename A3>
struct Func3
{
    Func3(H const& h, A1 const& a1, A2 const& a2, A3 const& a3) : h(h), a1(a1), a2(a2), a3(a3)
    {
    }
    void operator()()
    {
        h(a1, a2, a3);
    }
    void operator()() const
    {
        h(a1, a2, a3);
    }
    H h;
    A1 a1;
    A2 a2;
    A3 a3;
};

} // namespace detail

struct PostOp
{
    template <typename Service, typename Handler>
    static void execute(Service& io, Handler const& handler)
    {
        io.post(handler);
    }
};

struct DispatchOp
{
    template <typename Service, typename Handler>
    static void execute(Service& io, Handler const& handler)
    {
        io.dispatch(handler);
    }
};

template <typename Service, typename Handler, typename Op>
class requeue_handler_helper
{
public:
    typedef void result_type;

    typedef Service service_type;
    typedef Handler handler_type;

    requeue_handler_helper(service_type& service, handler_type h) : service_(service), handler_(h)
    {
    }

    void operator()()
    {
        service_.post(handler_);
    }
    void operator()() const
    {
        service_.post(handler_);
    }

    template <typename A0>
    void operator()(A0 a0)
    {
        service_.post(detail::Func1<handler_type, A0>(handler_, a0));
    }

    template <typename A0>
    void operator()(A0 a0) const
    {
        service_.post(detail::Func1<handler_type, A0>(handler_, a0));
    }

    template <typename A0, typename A1>
    void operator()(A0 a0, A1 a1)
    {
        service_.post(detail::Func2<handler_type, A0, A1>(handler_, a0, a1));
    }

    template <typename A0, typename A1>
    void operator()(A0 a0, A1 a1) const
    {
        service_.post(detail::Func2<handler_type, A0, A1>(handler_, a0, a1));
    }

    template <typename A0, typename A1, typename A2>
    void operator()(A0 a0, A1 a1, A2 a2)
    {
        service_.post(service_, detail::Func3<handler_type, A0, A1, A2>(handler_, a0, a1, a2));
    }

    template <typename A0, typename A1, typename A2>
    void operator()(A0 a0, A1 a1, A2 a2) const
    {
        service_.post(service_, detail::Func3<handler_type, A0, A1, A2>(handler_, a0, a1, a2));
    }

private:
    service_type& service_;
    handler_type handler_;
};

template <typename Op, typename Service, typename Handler>
requeue_handler_helper<Service, Handler, Op> requeue_handler(Service& service, Handler h)
{
    return requeue_handler_helper<Service, Handler, Op>(service, h);
}

#if DEBUG_PRINT
struct detail_printer
{
    static boost::mutex& mux()
    {
        static boost::mutex m;
        return m;
    }
};
#endif // DEBUG_PRINT

// Same as strand.wrap, but with well defined type
// (strand.wrap returns implementation specific type)
template <typename Strand, typename Handler, typename Op>
class strand_wrap_helper
{
public:
    typedef Strand strand_type;
    typedef Handler handler_type;

protected:
#if DEBUG_PRINT
    // for debugging - prints strand enters and exits
    template <typename H>
    struct strand_gate
    {
        typedef void result_type;

        struct printer
        {
            typedef boost::lock_guard<boost::mutex> lock_t;

            printer(strand_type* s) : s(s)
            {
                lock_t lock(detail_printer::mux());
                std::cout << "strand (" << (void*)s << ") >>> enter\n";
                std::cout.flush();
            }

            ~printer()
            {
                lock_t lock(detail_printer::mux());
                std::cout << "strand (" << (void*)s << ") <<< leave\n";
                std::cout.flush();
            }

            strand_type* s;
        };

        strand_type* s;
        H handler;
        strand_gate(strand_type* s, H const& handler) : s(s), handler(handler)
        {
        }

        void operator()()
        {
            printer prn(s);
            handler();
        }

        template <typename A0>
        void operator()(A0 a0)
        {
            printer prn(s);
            handler(a0);
        }

        template <typename A0, typename A1>
        void operator()(A0 a0, A1 a1)
        {
            printer prn(s);
            handler(a0, a1);
        }

        template <typename A0, typename A1, typename A2>
        void operator()(A0 a0, A1 a1, A2 a2)
        {
            printer prn(s);
            handler(a0, a1, a2);
        }

        void operator()() const
        {
            printer prn(s);
            handler();
        }

        template <typename A0>
        void operator()(A0 a0) const
        {
            printer prn(s);
            handler(a0);
        }

        template <typename A0, typename A1>
        void operator()(A0 a0, A1 a1) const
        {
            printer prn(s);
            handler(a0, a1);
        }

        template <typename A0, typename A1, typename A2>
        void operator()(A0 a0, A1 a1, A2 a2) const
        {
            printer prn(s);
            handler(a0, a1, a2);
        }
    };

    template <typename H>
    strand_gate<H> make_strand_gate(strand_type* s, H const& handler)
    {
        return strand_gate<H>(s, handler);
    }

    strand_gate<requeue_handler_helper<boost::asio::io_service, handler_type, Op>>
#else
    requeue_handler_helper<boost::asio::io_service, handler_type, Op>
#endif // DEBUG_PRINT
    requeue_handler_i()
    {
#if DEBUG_PRINT
        return make_strand_gate(
            &strand_,
            requeue_handler_helper<boost::asio::io_service, handler_type, Op>(
                strand_.get_io_service(), handler_));
#else
        return requeue_handler_helper<boost::asio::io_service, handler_type, Op>(
            strand_.get_io_service(), handler_);
#endif
    }

public:
    inline strand_type& strand() const
    {
        return strand();
    }

    strand_wrap_helper(strand_type& s, handler_type const& h) : strand_(s), handler_(h)
    {
    }

    void operator()()
    {
        Op::execute(strand_, requeue_handler_i());
    }

    void operator()() const
    {
        Op::execute(strand_, requeue_handler_i());
    }

    template <typename A0>
    void operator()(A0 a0)
    {
        Op::execute(strand_, boost::bind(requeue_handler_i(), a0));
    }

    template <typename A0>
    void operator()(A0 a0) const
    {
        Op::execute(strand_, boost::bind(requeue_handler_i(), a0));
    }

    template <typename A0, typename A1>
    void operator()(A0 a0, A1 a1)
    {
        Op::execute(strand_, boost::bind(requeue_handler_i(), a0, a1));
    }

    template <typename A0, typename A1>
    void operator()(A0 a0, A1 a1) const
    {
        Op::execute(strand_, boost::bind(requeue_handler_i(), a0, a1));
    }

    template <typename A0, typename A1, typename A2>
    void operator()(A0 a0, A1 a1, A2 a2)
    {
        Op::execute(strand_, boost::bind(requeue_handler_i(), a0, a1, a2));
    }

    template <typename A0, typename A1, typename A2>
    void operator()(A0 a0, A1 a1, A2 a2) const
    {
        Op::execute(strand_, boost::bind(requeue_handler_i(), a0, a1, a2));
    }

private:
    strand_type& strand_;
    handler_type handler_;
};

template <typename Strand, typename Handler>
strand_wrap_helper<Strand, Handler, DispatchOp> strand_wrap_dispatch(
    Strand& strand,
    Handler const& handler)
{
    return strand_wrap_helper<Strand, Handler, DispatchOp>(strand, handler);
}

template <typename Strand, typename Handler>
strand_wrap_helper<Strand, Handler, PostOp> strand_wrap_post(Strand& strand, Handler const& handler)
{
    return strand_wrap_helper<Strand, Handler, PostOp>(strand, handler);
}

template <typename Function, typename Strand, typename Handler, typename Op>
inline void asio_handler_invoke(
    const Function& function,
    strand_wrap_helper<Strand, Handler, Op>* this_handler)
{
    this_handler->strand().dispatch(function);
    // this_handler->strand ().get_io_service ().dispatch (function);
}

}} // namespace yplatform::net
