#pragma once

#include <yplatform/log.h>
#include <yplatform/net/dns/resolver.h>
#include <yplatform/net/types.h>
#include <yplatform/net/universal_socket.h>
#include <yplatform/net/handlers/mem_alloc.h>
#include <yplatform/net/handlers/protect_wrapper.h>
#include <yplatform/time_traits.h>

#define PROTECTED_TYPE(Handler)                                                                    \
    detail::protect_wrapper<typename std::remove_reference<Handler>::type>

namespace yplatform { namespace net {

namespace ph = std::placeholders;

enum address_family
{
    ipv4,
    ipv6
};

namespace detail {
template <address_family, typename Resolver>
struct resolve_step_helper;

template <typename Resolver>
struct resolve_step_helper<ipv4, Resolver>
{
    template <typename Handler>
    static void resolve(Resolver& resolver, const std::string& hostname, Handler&& handler)
    {
        resolver.async_resolve_a(hostname, std::forward<Handler>(handler));
    }
    typedef typename Resolver::iterator_a iterator;
};

template <typename Resolver>
struct resolve_step_helper<ipv6, Resolver>
{
    template <typename Handler>
    static void resolve(Resolver& resolver, const std::string& hostname, Handler&& handler)
    {
        resolver.async_resolve_aaaa(hostname, std::forward<Handler>(handler));
    }
    typedef typename Resolver::iterator_aaaa iterator;
};
}

template <typename Socket, typename Resolver = dns::resolver>
class async_sequental_connect_op
{
public:
    using socket_t = Socket;
    using endpoint_t = typename socket_t::endpoint_t;
    using resolver_t = Resolver;
    using iterator_ipv4 = typename resolver_t::iterator_a;
    using iterator_ipv6 = typename resolver_t::iterator_aaaa;
    using logger_t = yplatform::log::source;
    using time_point = time_traits::time_point;
    using duration = time_traits::duration;

    async_sequental_connect_op(
        socket_t& socket,
        client_settings::resolve_order_t resolve_order = client_settings::ipv6_ipv4)
        : socket_(socket)
        , resolver_(*socket_.get_io())
        , opcontext_(*socket_.get_io())
        , resolve_order_(resolve_order)
    {
    }

    time_point now()
    {
        return time_traits::clock::now();
    }

    void logger(const logger_t& new_logger)
    {
        logger_ = new_logger;
    }

    void set_log_prefix(const std::string& val)
    {
        logger_.set_log_prefix(val);
    }

    template <typename Handler>
    void perform(
        const string& hostname,
        unsigned short port,
        const duration& attempt_tm,
        const time_point& deadline,
        Handler&& handler)
    {
        attempt_timeout_ = attempt_tm;
        total_resolve_time_ = time_traits::duration::zero();
        deadline_ = deadline;
        connect_protected(hostname, port, protect_handler(std::forward<Handler>(handler)));
    }

    template <typename Handler>
    void perform(
        const string& hostname,
        unsigned short port,
        const duration& attempt_tm,
        const duration& total_tm,
        Handler&& handler)
    {
        perform(hostname, port, attempt_tm, now() + total_tm, std::forward<Handler>(handler));
    }

    time_point attempt_deadline()
    {
        return std::min(deadline_, now() + attempt_timeout_);
    }

    const duration& total_resolve_time() const
    {
        return total_resolve_time_;
    }

private:
    template <typename Handler>
    void connect_protected(const string& hostname, unsigned short port, Handler&& handler)
    {
        error_code ec;
        auto ip = boost::asio::ip::address::from_string(hostname, ec);
        if (ec)
        {
            switch (resolve_order_)
            {
            case client_settings::ipv4:
                do_one_step_connect<ipv4>(hostname, port, std::forward<Handler>(handler));
                break;
            case client_settings::ipv6:
                do_one_step_connect<ipv6>(hostname, port, std::forward<Handler>(handler));
                break;
            case client_settings::ipv4_ipv6:
                do_two_step_connect<ipv4, ipv6>(hostname, port, std::forward<Handler>(handler));
                break;
            default: // ipv6_ipv4 or enum has been modified
                do_two_step_connect<ipv6, ipv4>(hostname, port, std::forward<Handler>(handler));
            };
        }
        else
        {
            endpoint_t endpoint(ip, port);
            socket_.async_connect(endpoint, deadline_, std::forward<Handler>(handler));
        }
    }

    template <address_family af, typename Handler>
    void do_one_step_connect(const string& hostname, unsigned short port, Handler&& handler)
    {
        typedef detail::resolve_step_helper<af, resolver_t> step_t;

        opcontext_.prepare(socket_, deadline_);
        using handler_t = typename std::decay<Handler>::type;
        // It's useless to create alloc_handler here because
        // yplatform dns resolver service (as a wrapper for asio)
        // will rewrap it (for caching).
        step_t::resolve(
            resolver_,
            hostname,
            std::bind(
                &async_sequental_connect_op::handle_resolve<typename step_t::iterator, handler_t>,
                this,
                ph::_1,
                ph::_2,
                std::forward<Handler>(handler),
                port,
                hostname,
                now()));
    }

    template <address_family af1, address_family af2, typename Handler>
    void do_two_step_connect(const string& hostname, unsigned short port, Handler&& handler)
    {
        using handler_t = typename std::decay<Handler>::type;
        do_one_step_connect<af1>(
            hostname,
            port,
            [this, hostname, port, handler = handler_t(std::forward<Handler>(handler))](
                const error_code& e) {
                if (!e || (e == boost::asio::error::operation_aborted && now() >= deadline_))
                {
                    socket_.get_io()->post(std::bind(handler, e));
                }
                else
                {
                    this->do_one_step_connect<af2>(hostname, port, std::move(handler));
                }
            });
    }

    template <typename Iterator, typename Handler>
    void handle_resolve(
        const error_code& e,
        Iterator i,
        Handler& handler,
        unsigned short port,
        const string& hostname,
        const time_point& start)
    {
        total_resolve_time_ += now() - start;
        opcontext_.complete();
        if (e)
        {
            YLOG(logger_, warning) << "dns resolve failed: " << e.message()
                                   << ", conn=" << socket_.id() << " hostname=" << hostname;
            socket_.get_io()->post(std::bind(handler, e));
        }
        else if (empty(i))
        {
            YLOG(logger_, warning) << "dns resolve failed: empty resolve list"
                                   << ", conn=" << socket_.id() << " hostname=" << hostname;
            socket_.get_io()->post(std::bind(handler, e));
        }
        else
        {
            endpoint_t point(boost::asio::ip::address::from_string(*i), port);
            socket_.async_connect(
                point,
                attempt_deadline(),
                std::bind(
                    &async_sequental_connect_op::handle_connect<Iterator, Handler>,
                    this,
                    ph::_1,
                    i,
                    std::move(handler),
                    port));
        }
    }

    template <typename Iterator, typename Handler>
    void handle_connect(const error_code& e, Iterator i, Handler& handler, unsigned short port)
    {
        if (!e || (e == boost::asio::error::operation_aborted && now() >= deadline_))
        {
            socket_.get_io()->post(std::bind(handler, e));
            return;
        }

        YLOG(logger_, info) << "connect failed: " << e.message() << ", conn=" << socket_.id()
                            << " ip=" << *i << " : " << port << ", trying next";
        i++;
        if (!empty(i))
        {
            socket_.close();
            endpoint_t point(boost::asio::ip::address::from_string(*i), port);
            socket_.async_connect(
                point,
                attempt_deadline(),
                std::bind(
                    &async_sequental_connect_op::handle_connect<Iterator, Handler>,
                    this,
                    ph::_1,
                    i,
                    std::move(handler),
                    port));
        }
        else
        {
            socket_.close();
            socket_.get_io()->post(std::bind(handler, e));
        }
    }

    template <typename Iterator>
    bool empty(const Iterator& iterator)
    {
        return iterator == Iterator();
    }

    socket_t& socket_;
    resolver_t resolver_;
    operation_context opcontext_; // TODO bad name
    logger_t logger_;
    duration attempt_timeout_;
    time_point deadline_;
    duration total_resolve_time_;
    client_settings::resolve_order_t resolve_order_;
};

}}

#undef PROTECTED_TYPE
