#pragma once

#include <yplatform/api.h>
#include <yplatform/net/config.h>
#include <yplatform/net/io_data.h>
#include <yplatform/net/net_service.h>
#include <yplatform/net/acceptor.h>
#include <yplatform/net/handlers/mem_alloc.h>
#include <yplatform/net/socket.h>
#include <yplatform/reactor.h>
#include <boost/asio/yield.hpp>
#include <boost/thread/locks.hpp>
#include <boost/thread/shared_mutex.hpp>

namespace yplatform { namespace net {

struct alloc_data
{
    handler_allocator<512> allocator;
};

template <
    typename Session,
    typename StatsCont = service_stats,
    typename Acceptor = acceptor<typename Session::protocol_t, typename Session::settings_t>>
class [[deprecated]] server : public net_service<Session, StatsCont>
{
public:
    typedef Acceptor acceptor_t;
    typedef typename Session::settings_t session_settings_t;
    typedef boost::shared_ptr<acceptor_t> acceptor_ptr;
    typedef thread_safe_set<acceptor_ptr> acceptor_set;
    typedef boost::shared_ptr<Session> session_ptr;
    typedef typename Session::protocol_t protocol_t;
    typedef typename protocol_t::endpoint endpoint_t;
    typedef net_service<Session> base_t;

    server(const ::yplatform::reactor_ptr& reactor);
    virtual ~server();

    bool listen(const session_settings_t& settings, const string& addr, unsigned short port);

    void stop_server();
    void abort_server();

    const acceptor_set& acceptors() const
    {
        return acceptors_;
    }

    acceptor_set& acceptors()
    {
        return acceptors_;
    }

    boost::asio::io_service* get_io()
    {
        return this->get_io_pool().get_io_data()->get_io();
    }

private:
    bool create_acceptor(
        const session_settings_t& settings,
        const string& addr,
        unsigned short port,
        const io_pool_ptr& accept_reactor);

    void create_session(
        acceptor_ptr, boost::shared_ptr<endpoint_t>, boost::shared_ptr<alloc_data> alloc_data);

    void handle_accept(
        acceptor_ptr acceptor,
        boost::shared_ptr<endpoint_t>,
        session_ptr session,
        boost::shared_ptr<alloc_data> alloc_data,
        const boost::system::error_code& e);

    void open_session(session_ptr session, const endpoint_t&);

    typedef boost::shared_mutex mutex_t;
    typedef boost::shared_lock<mutex_t> read_lock;
    typedef boost::unique_lock<mutex_t> write_lock;

    acceptor_set acceptors_;
    bool stopped_;
    mutex_t mux_;
};

template <typename Session, typename StatsCont, typename Acceptor>
server<Session, StatsCont, Acceptor>::server(const ::yplatform::reactor_ptr& reactor)
    : base_t(reactor), stopped_(false)
{
}

template <typename Session, typename StatsCont, typename Acceptor>
server<Session, StatsCont, Acceptor>::~server()
{
}

template <typename Endpoint, typename Settings>
inline bool ipv6only(const Endpoint& ep, const Settings& settings)
{
    return (ep.protocol() == boost::asio::ip::tcp::v6() && settings.ipv6_only);
}

template <typename Session, typename StatsCont, typename Acceptor>
bool server<Session, StatsCont, Acceptor>::listen(
    const session_settings_t& settings,
    const string& addr,
    unsigned short port)
{
    for (const auto& io_pool : this->get_io_pool().reactor()->get_pools())
    {
        if (!create_acceptor(settings, addr, port, io_pool))
        {
            return false;
        }
    }
    return true;
}

template <typename Session, typename StatsCont, typename Acceptor>
bool server<Session, StatsCont, Acceptor>::create_acceptor(
    const session_settings_t& settings,
    const string& addr,
    unsigned short port,
    const io_pool_ptr& io_pool)
{
    acceptor_ptr acceptor = boost::make_shared<acceptor_t>(io_pool->io(), settings);
    acceptor->logger(this->logger());
    endpoint_t ep;
    typename protocol_t::resolver resolver(*io_pool->io());
    typename protocol_t::resolver::query query(addr, std::to_string(port));
    error_code ec;
    auto iterator = resolver.resolve(query, ec);
    if (ec)
    {
        error_code cast_ec;
        auto ip_addr = boost::asio::ip::address::from_string(addr, cast_ec);
        if (cast_ec)
        {
            YLOG_LOCAL(error) << "cannot resolve " << addr << ": " << ec.message();
            return false;
        }
        ep = endpoint_t(ip_addr, port);
    }
    else
    {
        ep = *iterator;
    }

    try
    {
        acceptor->open(ep.protocol());
        if (ipv6only(ep, settings))
        {
            acceptor->set_option(boost::asio::ip::v6_only(true));
        }
        acceptor->set_option(typename acceptor_t::reuse_address(true));
#ifdef SO_REUSEPORT // this flag is not supported in libc for kernels < 3.9 (e.g. in arcadia)
        if (settings.reuse_port)
        {
            const int reuse = 1;
            if (::setsockopt(
                    acceptor->native_handle(), SOL_SOCKET, SO_REUSEPORT, &reuse, sizeof(reuse)) < 0)
            {
                std::ostringstream stream;
                stream << "setsockopt SOL_SOCKET SO_REUSEPORT " << reuse
                       << " failed: " << strerror(errno);
                throw std::runtime_error(stream.str());
            }
        }
#endif

#ifdef SOL_TCP
        if (settings.defer_accept)
        {
            const auto timeout = settings.read_timeout.count();
            ::setsockopt(
                acceptor->native_handle(), SOL_TCP, TCP_DEFER_ACCEPT, &timeout, sizeof(timeout));
        }
#endif
        acceptor->bind(ep);
        acceptor->listen(settings.listen_backlog);
        acceptors_.insert(acceptor);
        this->create_session(
            acceptor, boost::make_shared<endpoint_t>(), boost::make_shared<alloc_data>());
        YLOG_LOCAL(info) << "starting listener on " << ep
                         << (ipv6only(ep, settings) ? " (ipv6 only)" : "");
    }
    catch (const std::exception& e)
    {
        YLOG_LOCAL(error) << "cannot start listener [" << ep << "]: " << e.what();
        return false;
    }
    return true;
}

template <typename Session, typename StatsCont, typename Acceptor>
void server<Session, StatsCont, Acceptor>::stop_server()
{
    write_lock lock(mux_);
    stopped_ = true;
    // cancel listen and wait while all sessions close
    typename thread_safe_set<acceptor_ptr>::uniq_lock_t lck = acceptors_.get_lock();
    for (typename std::set<acceptor_ptr>::iterator i = acceptors_.get_set().begin(),
                                                   i_end = acceptors_.get_set().end();
         i != i_end;
         ++i)
    {
        boost::system::error_code e;
        (*i)->cancel(e);
        (*i)->close(e);
    }
}

template <typename Session, typename StatsCont, typename Acceptor>
void server<Session, StatsCont, Acceptor>::abort_server()
{
    write_lock lock(mux_);
    acceptors_.get_set().clear();
}

template <typename Session, typename StatsCont, typename Acceptor>
void server<Session, StatsCont, Acceptor>::create_session(
    acceptor_ptr acceptor,
    boost::shared_ptr<endpoint_t> ep,
    boost::shared_ptr<alloc_data> alloc_data)
{
    session_ptr sess =
        boost::make_shared<Session>(this->get_io_pool().get_io_data(), acceptor->settings());
    sess->logger(this->logger());
    read_lock lock(mux_);
    if (stopped_) return;

    acceptor->async_accept(
        sess->raw_socket(),
        *ep,
        make_custom_alloc_handler(
            alloc_data->allocator,
            bind(
                &server::handle_accept,
                this,
                acceptor,
                ep,
                sess,
                alloc_data,
                boost::asio::placeholders::error)));
}

template <typename Session, typename StatsCont, typename Acceptor>
void server<Session, StatsCont, Acceptor>::handle_accept(
    acceptor_ptr acceptor,
    boost::shared_ptr<endpoint_t> ep,
    session_ptr session,
    boost::shared_ptr<alloc_data> alloc_data,
    const boost::system::error_code& e)
{
    if (stopped_) return;

    auto ep_copy = *ep;

    if (e != boost::asio::error::operation_aborted) this->create_session(acceptor, ep, alloc_data);

    if (!e)
    {
        session->get_io()->post(make_custom_alloc_handler(
            session->io_ops_allocator(),
            boost::bind(&server::open_session, this, session, ep_copy)));
    }
    else if (e == boost::asio::error::operation_aborted)
    {
        YLOG_CTX_LOCAL(session->get_context(), error) << "server::handle_accept: stopped";
    }
    else
    {
        YLOG_CTX_LOCAL(session->get_context(), error)
            << "server::handle_accept: error: " << e.message();
    }
}

template <typename Session, typename StatsCont, typename Acceptor>
void server<Session, StatsCont, Acceptor>::open_session(session_ptr session, const endpoint_t& ep)
{
    boost::system::error_code ec;
    session->open(ec);
    if (ec)
    {
        YLOG_CTX_LOCAL(session->get_context(), error)
            << "server::open_session error: " << ec.message() << " src=" << ep.address() << ":"
            << ep.port();
        return;
    }
    this->core_register_session(session);
}

template <typename Acceptor>
class basic_server
    : public boost::asio::coroutine
    , public yplatform::log::contains_logger
{
public:
    using socket_type = typename Acceptor::socket_type;
    using endpoint_type = typename Acceptor::endpoint_type;
    using accept_handler_type = std::function<void(socket_type&&)>;

    template <typename Dummy = void>
    basic_server(
        io_data& io_data,
        const std::string& path,
        const settings& st = settings(),
        typename std::enable_if<std::is_same<socket_type, unix_socket>::value, Dummy>::type* = 0)
        : io_data_(io_data)
        , acceptor_(new Acceptor(*io_data_.get_io(), endpoint_type(path)))
        , settings_(st)
    {
    }

    template <typename Dummy = void>
    basic_server(
        io_data& io_data,
        const string& addr,
        unsigned short port,
        const server_settings& settings = server_settings(),
        typename std::enable_if<std::is_same<socket_type, tcp_socket>::value, Dummy>::type* = 0)
        : io_data_(io_data), acceptor_(new Acceptor(*io_data_.get_io())), settings_(settings)
    {
        boost::system::error_code ec;
        auto endpoint = resolve(addr, port, ec);
        if (ec) throw std::runtime_error("server init failed with " + ec.message());
        acceptor_->bind(endpoint, settings);
    }

    template <typename Handler>
    void listen(Handler&& handler)
    {
        socket_ = std::make_shared<socket_type>(io_data_, settings_);
        handler_ = std::forward<Handler>(handler);
        (*this)();
    }

    void stop()
    {
        acceptor_->close();
    }

    void operator()(boost::system::error_code err = boost::system::error_code())
    {
        reenter(this)
        {
            while (acceptor_->is_open())
            {
                *ep_ = endpoint_type{};
                yield acceptor_->async_accept(*socket_, *ep_, *this);
                try
                {
                    if (err == boost::asio::error::bad_descriptor ||
                        err == boost::asio::error::operation_aborted)
                    {
                        YLOG_G(info) << "accept stop, error: " << err.message();
                        return;
                    }
                    if (!err) socket_->open(err);
                    if (!err)
                    {
                        handler_(std::move(*socket_));
                    }
                    else
                    {
                        YLOG_G(error) << "accept error: " << err.message()
                                      << " src=" << ep_->address() << ":" << ep_->port();
                        socket_->close();
                    }
                }
                catch (const std::exception& e)
                {
                    try
                    {
                        YLOG_G(error) << "accept exception: " << e.what();
                    }
                    catch (...)
                    {
                    }
                }
                catch (...)
                {
                    try
                    {
                        YLOG_G(error) << "accept unknown exception";
                    }
                    catch (...)
                    {
                    }
                }
                *socket_ = socket_type(io_data_, settings_);
            }
        }
    }

private:
    template <typename Dummy = void>
    endpoint_type resolve(
        const std::string& addr,
        unsigned short port,
        boost::system::error_code& ec,
        typename std::enable_if<std::is_same<socket_type, tcp_socket>::value, Dummy>::type* = 0)
    {
        using protocol_type = typename Acceptor::protocol_type;
        typename protocol_type::resolver resolver(*io_data_.get_io());
        typename protocol_type::resolver::query query(addr, std::to_string(port));
        auto iterator = resolver.resolve(query, ec);
        endpoint_type ep;
        if (ec)
        {
            auto ip_addr = boost::asio::ip::address::from_string(addr, ec);
            if (ec) return endpoint_type();
            ep = endpoint_type(ip_addr, port);
        }
        else
        {
            ep = *iterator;
        }
        return ep;
    }

    io_data& io_data_;
    std::shared_ptr<Acceptor> acceptor_;
    std::shared_ptr<socket_type> socket_;
    typename socket_type::settings_t settings_;
    accept_handler_type handler_;
    std::shared_ptr<endpoint_type> ep_{ new endpoint_type };
};

using tcp_server = basic_server<tcp_acceptor>;
using unix_socket_server = basic_server<unix_socket_acceptor>;

}}

#include <boost/asio/unyield.hpp>
