#pragma once

#include <yplatform/net/types.h>
#include <yplatform/net/io_data.h>
#include <yplatform/net/keep_alive.h>
#include <yplatform/net/context.h>
#include <yplatform/net/stream/optional_ssl_socket.h>
#include <yplatform/net/streamable.h>
#include <yplatform/net/buffer_sequence.h>
#include <yplatform/net/buffers/chunk.h>
#include <yplatform/net/handlers/mem_alloc.h>
#include <yplatform/net/handlers/protect_wrapper.h>
#include <yplatform/net/handlers/timer_handler.h>
#include <yplatform/application/task_context.h>
#include <yplatform/time_traits.h>
#include <yplatform/log.h>
#include <boost/asio.hpp>
#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/function.hpp>

#define NET_SESSION_LOG(severity)                                                                  \
    YLOG_CTX_LOCAL(this->get_context(), severity) << this->get_log_prefix() << " "

#define PROTECTED_TYPE(Handler) detail::protect_wrapper<Handler>

namespace yplatform { namespace net {

struct operation_context_t
{
    operation_context_t(boost::asio::io_service& io) : timer(io), active(false)
    {
    }

    time_traits::timer timer;
    bool active;
    handler_allocator<1024> allocator;
};

template <typename Settings, typename Protocol>
class session_strand : public yplatform::log::contains_logger
{
public:
    typedef Protocol protocol_t;
    typedef Settings settings_t;
    typedef typename protocol_t::socket raw_socket_t;
    typedef typename protocol_t::socket::endpoint_type endpoint_t;

    typedef boost::function<void(const error_code&)> close_hook_t;
    typedef boost::function<void()> shutdown_hook_t;
    typedef boost::function<void()> cancel_hook_t;
    typedef boost::function<void(const error_code&)> tls_hook_t;

    typedef stream::optional_ssl_socket<raw_socket_t> stream_t;

public:
    session_strand(
        io_data* io_data,
        const settings_t& settings,
        boost::asio::ssl::stream_base::handshake_type tls_handshake_type);

    virtual ~session_strand();

    error_code open(error_code& ec);

    // put shared_from_this in binded Handler
    template <typename Handler>
    void async_close(Handler&& handler)
    {
        strand_.dispatch(boost::bind(
            &session_strand::real_close<PROTECTED_TYPE(Handler)>,
            this,
            protect_handler(static_cast<Handler&&>(handler))));
    }

    // put shared_from_this in binded Handler
    template <typename Handler>
    void do_shutdown(Handler&& handler, bool graceful)
    {
        strand_.dispatch(boost::bind(
            &session_strand::real_do_shutdown<PROTECTED_TYPE(Handler)>,
            this,
            protect_handler(static_cast<Handler&&>(handler)),
            graceful));
    }

    // put shared_from_this in binded Handler
    template <typename Handler>
    void cancel_operations(Handler&& handler)
    {
        strand_.dispatch(boost::bind(
            &session_strand::real_cancel_operations<PROTECTED_TYPE(Handler)>,
            this,
            protect_handler(static_cast<Handler&&>(handler))));
    }

    void const* id() const
    {
        return static_cast<void const*>(this);
    }

    handler_allocator<128>& io_ops_allocator()
    {
        return io_ops_allocator_;
    }

    stream_t& stream()
    {
        return stream_;
    }

    const boost::asio::ip::address& remote_addr() const
    {
        return remote_addr_;
    }

    unsigned short remote_port() const
    {
        return remote_port_;
    }

    const boost::asio::ip::address& local_addr() const
    {
        return local_addr_;
    }

    unsigned short local_port() const
    {
        return local_port_;
    }

    virtual task_context_ptr get_context() const
    {
        return task_context_ptr();
    }

    boost::asio::io_service* get_io()
    {
        return io_data_->get_io();
    }

    const boost::asio::io_service* get_io() const
    {
        return io_data_->get_io();
    }

    raw_socket_t& raw_socket()
    {
        return raw_socket_;
    }

    const raw_socket_t& raw_socket() const
    {
        return raw_socket_;
    }

    boost::asio::io_service::strand& strand()
    {
        return strand_;
    }

    template <typename Handler>
    void start_tls(Handler&& handler)
    {
        this->raw_socket().cancel();
        stream().async_start_tls(
            *io_data_->get_ssl_context(),
            tls_handshake_type_,
            strand().wrap(make_custom_alloc_handler(
                tls_allocator_,
                boost::bind(
                    &session_strand::handle_tls_enabled<PROTECTED_TYPE(Handler)>,
                    this,
                    _1,
                    protect_handler(static_cast<Handler&&>(handler))))));
    }
    template <typename Handler, typename TimerHandler>
    void start_tls(Handler&& handler, TimerHandler&& deadline_handler)
    {
        tls_timer_.expires_from_now(this->settings().tls_timeout);
        tls_timer_.async_wait(
            strand().wrap(create_timer_handler(static_cast<TimerHandler&&>(deadline_handler))));
        this->raw_socket().cancel();
        stream().async_start_tls(
            *io_data_->get_ssl_context(),
            tls_handshake_type_,
            strand().wrap(make_custom_alloc_handler(
                tls_allocator_,
                boost::bind(
                    &session_strand::handle_tls_enabled<PROTECTED_TYPE(Handler)>,
                    this,
                    _1,
                    protect_handler(static_cast<Handler&&>(handler))))));
    }
    template <typename Handler, typename TimerHandler, typename VerifyHandler>
    void start_tls(
        Handler&& handler,
        TimerHandler&& deadline_handler,
        VerifyHandler&& verify_handler)
    {
        tls_timer_.expires_from_now(this->settings().tls_timeout);
        tls_timer_.async_wait(
            strand().wrap(create_timer_handler(static_cast<TimerHandler&&>(deadline_handler))));
        this->raw_socket().cancel();
        stream().async_start_tls(
            *io_data_->get_ssl_context(),
            tls_handshake_type_,
            strand().wrap(make_custom_alloc_handler(
                tls_allocator_,
                boost::bind(
                    &session_strand::handle_tls_enabled<PROTECTED_TYPE(Handler)>,
                    this,
                    _1,
                    protect_handler(static_cast<Handler&&>(handler))))),
            static_cast<VerifyHandler&&>(verify_handler));
    }

    template <typename RecvHandler, typename RecvBuffer>
    void async_read(RecvHandler&& handler, RecvBuffer&& buff, std::size_t min)
    {
        assert(!read_context_.active);
        read_context_.active = true;

        boost::asio::async_read(
            stream(),
            buff,
            boost::asio::transfer_at_least(min),
            strand().wrap(make_custom_alloc_handler(
                read_context_.allocator,
                boost::bind(
                    &session_strand::handle_read<PROTECTED_TYPE(RecvHandler)>,
                    this,
                    protect_handler(static_cast<RecvHandler&&>(handler)),
                    _1,
                    _2))));
    }

    template <typename RecvHandler, typename RecvBuffer>
    void async_read(RecvHandler&& handler, RecvBuffer&& buff)
    {
        assert(!read_context_.active);
        read_context_.active = true;

        boost::asio::async_read(
            stream(),
            buff,
            strand().wrap(make_custom_alloc_handler(
                read_context_.allocator,
                boost::bind(
                    &session_strand::handle_read<PROTECTED_TYPE(RecvHandler)>,
                    this,
                    protect_handler(static_cast<RecvHandler&&>(handler)),
                    _1,
                    _2))));
    }

    template <typename RecvHandler, typename DeadlineHandler, typename RecvBuffer>
    void async_read(
        RecvHandler&& handler,
        DeadlineHandler&& deadline_handler,
        const time_traits::duration& read_timeout,
        RecvBuffer&& buff,
        std::size_t min = 1U)
    {
        assert(!read_context_.active);
        read_context_.active = true;
        read_context_.timer.expires_from_now(read_timeout);
        read_context_.timer.async_wait(
            strand().wrap(create_timer_handler(static_cast<DeadlineHandler&&>(deadline_handler))));

        boost::asio::async_read(
            stream(),
            buff,
            boost::asio::transfer_at_least(min),
            strand().wrap(make_custom_alloc_handler(
                read_context_.allocator,
                boost::bind(
                    &session_strand::handle_read<PROTECTED_TYPE(RecvHandler)>,
                    this,
                    protect_handler(static_cast<RecvHandler&&>(handler)),
                    _1,
                    _2))));
    }

    template <typename RecvHandler, typename DeadlineHandler, typename RecvBuffer>
    void async_read(
        RecvHandler&& handler,
        DeadlineHandler&& deadline_handler,
        RecvBuffer&& buff,
        std::size_t min = 1U)
    {
        async_read(
            std::move(handler),
            std::move(deadline_handler),
            this->settings().read_timeout,
            std::move(buff),
            min);
    }

    template <typename RecvHandler, typename DeadlineHandler, typename RecvBuffer>
    void async_read(
        RecvHandler&& handler,
        DeadlineHandler&& deadline_handler,
        RecvBuffer&& buff,
        const time_traits::duration& read_timeout)
    {
        assert(!read_context_.active);
        read_context_.active = true;
        read_context_.timer.expires_from_now(read_timeout);
        read_context_.timer.async_wait(
            strand().wrap(create_timer_handler(static_cast<DeadlineHandler&&>(deadline_handler))));

        boost::asio::async_read(
            stream(),
            buff,
            strand().wrap(make_custom_alloc_handler(
                read_context_.allocator,
                boost::bind(
                    &session_strand::handle_read<PROTECTED_TYPE(RecvHandler)>,
                    this,
                    protect_handler(static_cast<RecvHandler&&>(handler)),
                    _1,
                    _2))));
    }

    template <typename RecvHandler, typename DeadlineHandler, typename RecvBuffer, typename Filter>
    void async_read_until(
        RecvHandler&& handler,
        DeadlineHandler&& deadline_handler,
        const time_traits::duration& read_timeout,
        RecvBuffer&& buff,
        const Filter& filter)
    {
        assert(!read_context_.active);
        read_context_.active = true;
        read_context_.timer.expires_from_now(read_timeout);
        read_context_.timer.async_wait(
            strand().wrap(create_timer_handler(static_cast<DeadlineHandler&&>(deadline_handler))));

        boost::asio::async_read_until(
            stream(),
            buff,
            filter,
            strand().wrap(make_custom_alloc_handler(
                read_context_.allocator,
                boost::bind(
                    &session_strand::handle_read<PROTECTED_TYPE(RecvHandler)>,
                    this,
                    protect_handler(static_cast<RecvHandler&&>(handler)),
                    _1,
                    _2))));
    }

    template <typename RecvHandler, typename DeadlineHandler, typename RecvBuffer, typename Filter>
    void async_read_until(
        RecvHandler&& handler,
        DeadlineHandler&& deadline_handler,
        RecvBuffer&& buff,
        const Filter& filter)
    {
        async_read_until(
            std::move(handler),
            std::move(deadline_handler),
            this->settings().read_timeout,
            std::move(buff),
            filter);
    }

    template <typename SendHandler, typename SendBufferPtr>
    void async_write_shared_buff(SendBufferPtr buff, SendHandler&& handler)
    {
        assert(!write_context_.active);
        write_context_.active = true;

        boost::asio::async_write(
            stream(),
            *buff,
            boost::asio::transfer_all(),
            strand().wrap(make_custom_alloc_handler(
                write_context_.allocator,
                boost::bind(
                    &session_strand::handle_write<PROTECTED_TYPE(SendHandler), SendBufferPtr>,
                    this,
                    protect_handler(static_cast<SendHandler&&>(handler)),
                    buff,
                    _1,
                    _2))));
    }

    template <typename SendHandler, typename SendBufferSeq>
    void async_write(const SendBufferSeq& buff, SendHandler&& handler)
    {
        assert(!write_context_.active);
        write_context_.active = true;

        boost::asio::async_write(
            stream(),
            buff,
            boost::asio::transfer_all(),
            strand().wrap(make_custom_alloc_handler(
                write_context_.allocator,
                boost::bind(
                    &session_strand::handle_write<PROTECTED_TYPE(SendHandler), SendBufferSeq>,
                    this,
                    protect_handler(static_cast<SendHandler&&>(handler)),
                    buff,
                    _1,
                    _2))));
    }

    template <typename SendHandler, typename DeadlineHandler, typename SendBufferPtr>
    void async_write_shared_buff(
        SendBufferPtr buff,
        SendHandler&& handler,
        DeadlineHandler&& deadline_handler,
        const time_traits::duration& write_timeout)
    {
        assert(!write_context_.active);
        write_context_.active = true;
        write_context_.timer.expires_from_now(write_timeout);
        write_context_.timer.async_wait(
            strand().wrap(create_timer_handler(static_cast<DeadlineHandler&&>(deadline_handler))));

        boost::asio::async_write(
            stream(),
            *buff,
            boost::asio::transfer_all(),
            strand().wrap(make_custom_alloc_handler(
                write_context_.allocator,
                boost::bind(
                    &session_strand::handle_write<PROTECTED_TYPE(SendHandler), SendBufferPtr>,
                    this,
                    protect_handler(static_cast<SendHandler&&>(handler)),
                    buff,
                    _1,
                    _2))));
    }

    template <typename SendHandler, typename DeadlineHandler, typename SendBufferPtr>
    void async_write_shared_buff(
        SendBufferPtr buff,
        SendHandler&& handler,
        DeadlineHandler&& deadline_handler)
    {
        async_write_shared_buff(
            buff, std::move(handler), std::move(deadline_handler), this->settings().write_timeout);
    }

    template <typename SendHandler, typename DeadlineHandler, typename SendBufferSeq>
    void async_write(
        const SendBufferSeq& buff,
        SendHandler&& handler,
        DeadlineHandler&& deadline_handler,
        const time_traits::duration& write_timeout)
    {
        assert(!write_context_.active);
        write_context_.active = true;
        write_context_.timer.expires_from_now(write_timeout);
        write_context_.timer.async_wait(
            strand().wrap(create_timer_handler(static_cast<DeadlineHandler&&>(deadline_handler))));

        boost::asio::async_write(
            stream(),
            buff,
            boost::asio::transfer_all(),
            strand().wrap(make_custom_alloc_handler(
                write_context_.allocator,
                boost::bind(
                    &session_strand::handle_write<PROTECTED_TYPE(SendHandler), SendBufferSeq>,
                    this,
                    protect_handler(static_cast<SendHandler&&>(handler)),
                    buff,
                    _1,
                    _2))));
    }

    template <typename SendHandler, typename DeadlineHandler, typename SendBufferSeq>
    void async_write(
        const SendBufferSeq& buff,
        SendHandler&& handler,
        DeadlineHandler&& deadline_handler)
    {
        async_write(
            buff, std::move(handler), std::move(deadline_handler), this->settings().write_timeout);
    }

protected:
    virtual void core_open()
    {
    }

    template <typename Handler>
    void handle_tls_enabled(const error_code& e, const Handler& handler)
    {
        tls_timer_.cancel();
        if (e)
        {
            NET_SESSION_LOG(notice) << "handle_tls_enabled: error: " << e.message();
        }
        else
        {
            NET_SESSION_LOG(debug) << "handle_tls_enabled: ok";
        }
        static const bool SSLKEYLOG = std::getenv(SSLKEYLOG_ENV) != nullptr;
        if (SSLKEYLOG)
        {
            auto sslkey = stream_.tap_ssl_key();
            if (sslkey.size()) YLOG_G(info) << sslkey;
        }
        handler(e);
    }

    template <typename Handler>
    void handle_tls_disabled(const error_code& e, const Handler& handler)
    {
        tls_timer_.cancel();
        if (e)
        {
            NET_SESSION_LOG(notice) << "handle_tls_disabled: error: " << e.message();
        }
        else
        {
            NET_SESSION_LOG(debug) << "handle_tls_disabled: ok";
        }
        handler(e);
    }

    const settings_t& settings() const
    {
        return (local_settings_ ? *local_settings_ : settings_);
    }

    settings_t& local_settings()
    {
        if (!local_settings_)
        {
            local_settings_ = new settings_t(settings_);
        }
        return *local_settings_;
    }

    template <typename RecvHandler>
    void handle_read(const RecvHandler& handler, const error_code& e, std::size_t bytes)
    {
        assert(read_context_.active);
        read_context_.timer.cancel();
        read_context_.active = false;
        handler(e, bytes);
    }

    template <typename SendHandler, typename SendBuffer>
    inline void handle_write(
        const SendHandler& handler,
        const SendBuffer& buff,
        const error_code& e,
        std::size_t bytes)
    {
        assert(write_context_.active);
        write_context_.timer.cancel();
        write_context_.active = false;
        handler(e, bytes, buff);
    }

    virtual const string& get_log_prefix() const
    {
        return log_prefix_;
    }

    void set_log_prefix(const string& log_prefix)
    {
        log_prefix_ = log_prefix;
    }

    void cancel_operations()
    {
        assert(strand().running_in_this_thread());
        real_cancel_operations();
    }

private:
    template <typename Handler>
    void real_close(const Handler& handler)
    {
        boost::system::error_code ec;
        if (stream().is_open())
        {
            stream().close(ec);
        }
        handler(ec);
    }

    template <typename Handler>
    void real_cancel_operations(const Handler& handler)
    {
        real_cancel_operations();
        handler();
    }

    void real_cancel_operations()
    {
        if (stream().is_open())
        {
            stream().cancel();
        }
    }

    template <typename Handler>
    void real_do_shutdown(const Handler& handler, bool graceful);

    io_data* io_data_;
    const settings_t& settings_;
    settings_t* local_settings_;

    raw_socket_t raw_socket_;
    stream_t stream_;
    boost::asio::io_service::strand strand_;
    string log_prefix_;
    boost::asio::ssl::stream_base::handshake_type tls_handshake_type_;

    boost::asio::ip::address remote_addr_;
    unsigned short remote_port_;
    boost::asio::ip::address local_addr_;
    unsigned short local_port_;

    operation_context_t read_context_;
    operation_context_t write_context_;
    handler_allocator<512> tls_allocator_;
    time_traits::timer tls_timer_;
    handler_allocator<128> io_ops_allocator_;
};

template <typename S, typename P>
session_strand<S, P>::session_strand(
    io_data* io_data,
    const settings_t& settings,
    boost::asio::ssl::stream_base::handshake_type tls_handshake_type)
    : io_data_(io_data)
    , settings_(settings)
    , local_settings_(0)
    , raw_socket_(*io_data->get_io())
    , stream_(raw_socket_)
    , strand_(*io_data->get_io())
    , tls_handshake_type_(tls_handshake_type)
    , remote_port_(0)
    , local_port_(0)
    , read_context_(*io_data->get_io())
    , write_context_(*io_data->get_io())
    , tls_allocator_()
    , tls_timer_(*io_data->get_io())
{
}

template <typename S, typename P>
session_strand<S, P>::~session_strand()
{
    delete local_settings_;
}

template <typename S, typename P>
error_code session_strand<S, P>::open(error_code& ec)
{
    ec = error_code();
    typename raw_socket_t::endpoint_type ep = this->raw_socket().remote_endpoint(ec);
    if (ec) return ec;
    remote_addr_ = ep.address();
    remote_port_ = ep.port();
    ep = this->raw_socket().local_endpoint(ec);
    if (ec) return ec;
    local_addr_ = ep.address();
    local_port_ = ep.port();
    setup_keep_alive(this->raw_socket(), this->settings());
    if (settings_.tcp_no_delay)
    {
        this->raw_socket().set_option(
            boost::asio::detail::socket_option::boolean<IPPROTO_TCP, TCP_NODELAY>(true));
    }
    if (settings_.tcp_quickack)
    {
        this->raw_socket().set_option(
            boost::asio::detail::socket_option::boolean<IPPROTO_TCP, TCP_QUICKACK>(true));
    }

    if (this->get_context())
    {
        NET_SESSION_LOG(debug) << "new session opened " << remote_addr_.to_string() << ":"
                               << remote_port_ << " -> " << local_addr_.to_string() << ":"
                               << local_port_;
    }

    //  strand().dispatch(boost::bind(&session_strand::core_open, this));
    core_open();
    return ec;
}

template <typename S, typename P>
template <typename Handler>
void session_strand<S, P>::real_do_shutdown(const Handler& handler, bool graceful)
{
    try
    {
        if (graceful)
        {
            stream().shutdown(raw_socket_t::shutdown_send);
        }
        else
        {
            stream().cancel();
            stream().close();
        }
        handler();
    }
    catch (std::exception const& err)
    {
        NET_SESSION_LOG(error) << "do_shutdown: std::exception: " << err.what();
    }
    catch (...)
    {
        NET_SESSION_LOG(error) << "do_shutdown: unknown exception";
    }
}

}}

#undef PROTECTED_TYPE
