#pragma once

#include <yplatform/ptree.h>
#include <yplatform/net/types.h>
#include <yplatform/time_traits.h>

namespace yplatform { namespace net {

struct settings
{
    enum
    {
        KB = 1024,
        MB = 1024 * KB,
        GB = 1024 * MB
    };

    time_traits::duration write_timeout;
    time_traits::duration read_timeout;
    time_traits::duration tls_timeout;

    bool tcp_no_delay; // TCP_NODELAY
    bool tcp_quickack; // TCP_QUICKACK
    bool no_keep_alive;
    int tcp_keep_idle;
    int tcp_keep_intvl;
    int tcp_keep_cnt;
    int tcp_linger2;

    settings()
        : write_timeout(time_traits::duration::max())
        , read_timeout(time_traits::duration::max())
        , tls_timeout(time_traits::duration::max())
        , tcp_no_delay(false)
        // John Nagle:
        // Set TCP_QUICKACK. If you find a case where that makes things worse, let me know.
        , tcp_quickack(true)
        , no_keep_alive(false)
        , tcp_keep_idle(-1)
        , tcp_keep_intvl(-1)
        , tcp_keep_cnt(-1)
        , tcp_linger2(-1)
    {
    }

    virtual ~settings()
    {
    }

    virtual void parse_ptree(const yplatform::ptree& data)
    {
        int read_milsec = data.get("read_timeout", -1);
        int write_milsec = data.get("write_timeout", -1);
        int tls_milsec = data.get("tls_timeout", -1);
        if (read_milsec >= 0) read_timeout = time_traits::milliseconds(read_milsec);
        if (write_milsec >= 0) write_timeout = time_traits::milliseconds(write_milsec);
        if (tls_milsec >= 0) tls_timeout = time_traits::milliseconds(tls_milsec);

        tcp_no_delay = data.get("nodelay", tcp_no_delay);
        tcp_no_delay = data.get("no_delay", tcp_no_delay);    // alias
        tcp_no_delay = data.get("tcp_nodelay", tcp_no_delay); // alias
        tcp_quickack = data.get("quickack", tcp_quickack);
        tcp_quickack = data.get("tcp_quickack", tcp_quickack); // alias
        no_keep_alive = (data.get("keep_alive.enable", (no_keep_alive ? 0 : 1)) == 0);
        if (!no_keep_alive)
        {
            tcp_keep_idle = data.get("keep_alive.tcp_keep_idle", tcp_keep_idle);
            tcp_keep_intvl = data.get("keep_alive.tcp_keep_intvl", tcp_keep_intvl);
            tcp_keep_cnt = data.get("keep_alive.tcp_keep_cnt", tcp_keep_cnt);
            tcp_linger2 = data.get("keep_alive.tcp_linger2", tcp_linger2);
        }
    }
};

struct client_settings : public settings
{
    enum resolve_order_t
    {
        ipv4,
        ipv6,
        ipv4_ipv6,
        ipv6_ipv4
    };

    resolve_order_t resolve_order;
    time_traits::duration resolve_timeout;
    time_traits::duration connect_timeout;

    client_settings()
        : resolve_order(ipv6_ipv4)
        , resolve_timeout(time_traits::duration::max())
        , connect_timeout(time_traits::duration::max())
    {
    }

    virtual void parse_ptree(const yplatform::ptree& data)
    {
        std::string ro = data.get("resolve_order", "ipv6_ipv4");
        if (ro == "ipv4") resolve_order = ipv4;
        else if (ro == "ipv6")
            resolve_order = ipv6;
        else if (ro == "ipv4_ipv6")
            resolve_order = ipv4_ipv6;
        else if (ro == "ipv6_ipv4")
            resolve_order = ipv6_ipv4;

        int resolve_milsec = data.get("resolve_timeout", -1);
        int connect_milsec = data.get("connect_timeout", -1);
        if (resolve_milsec >= 0) resolve_timeout = time_traits::milliseconds(resolve_milsec);
        if (connect_milsec >= 0) connect_timeout = time_traits::milliseconds(connect_milsec);
        settings::parse_ptree(data);
    }
};

struct server_settings : public settings
{
    enum
    {
        DEFAULT_LISTEN_BACKLOG = SOMAXCONN
    };

    bool defer_accept = false;
    bool ipv6_only = false;
    int listen_backlog = DEFAULT_LISTEN_BACKLOG;
    bool reuse_port = true;

    virtual void parse_ptree(const yplatform::ptree& data)
    {
        defer_accept = (data.get("defer_accept", defer_accept ? 1 : 0) != 0);
        ipv6_only = (data.get("ipv6_only", ipv6_only ? 1 : 0) == 1);
        listen_backlog = data.get("listen_backlog", listen_backlog);
        reuse_port = data.get("reuse_port", reuse_port);
        if (listen_backlog < 0) listen_backlog = DEFAULT_LISTEN_BACKLOG;
        settings::parse_ptree(data);
    }
};

using acceptor_settings = server_settings;

}}
