#pragma once

#include <yplatform/util/private_access.h>
#include <boost/asio.hpp>
#include <boost/asio/ssl/stream.hpp>

namespace yplatform { namespace net { namespace stream { namespace detail {

template <typename Socket>
using ssl_stream = boost::asio::ssl::stream<Socket&>;
using ssl_tcp_stream = ssl_stream<boost::asio::ip::tcp::socket>;
using ssl_stream_core = boost::asio::ssl::detail::stream_core;
using ssl_stream_core_tag = private_access_tag<ssl_tcp_stream, ssl_stream_core>;
using ext_bio_tag = private_access_tag<boost::asio::ssl::detail::engine, BIO*>;

template <typename Socket>
inline ssl_stream_core* get_core(ssl_stream<Socket>&)
{
    return nullptr;
}

template <>
inline ssl_stream_core* get_core(ssl_tcp_stream& stream)
{
    return &get_field<ssl_stream_core_tag>(stream);
}

inline void reduce_buffer(
    const boost::asio::mutable_buffer& buf,
    std::vector<unsigned char>& buf_data,
    size_t buffer_size)
{
    buf_data.resize(buffer_size);
    buf_data.shrink_to_fit();
    boost::asio::mutable_buffer* mutable_buf = const_cast<boost::asio::mutable_buffer*>(&buf);
    *mutable_buf = boost::asio::mutable_buffer(boost::asio::buffer(buf_data));
}

inline void copy_data_from_bio(BIO* old_bio, BIO* new_bio, const boost::asio::mutable_buffer& buf)
{
    int length_recv = ::BIO_read(old_bio, buf.data(), static_cast<int>(buf.size()));
    int length_send = ::BIO_write(new_bio, buf.data(), length_recv);
    if (length_recv != length_send) throw std::runtime_error("failed to copy data from bio");
}

template <typename Socket>
inline void reduce_stream_core_buffers(ssl_stream<Socket>& stream, size_t buffer_size)
{
    auto core = get_core(stream);
    if (!core) return;
    // Input buffer can contain unconsumed data even if there are no active operations
    // on stream. So reduce it size only if data fits in new buffer.
    if (core->input_.size() <= buffer_size)
    {
        reduce_buffer(core->input_buffer_, core->input_buffer_space_, buffer_size);
        core->input_ = boost::asio::buffer(core->input_buffer_, core->input_.size());
    }
    reduce_buffer(core->output_buffer_, core->output_buffer_space_, buffer_size);
}

inline std::pair<BIO**, BIO*> extract_bio_pair(ssl_stream_core& core)
{
    auto ssl = core.engine_.native_handle();
    auto int_bio = SSL_get_rbio(ssl);
    auto& ext_bio = get_field<ext_bio_tag>(core.engine_);
    return { &ext_bio, int_bio };
}

inline std::pair<BIO*, BIO*> make_new_bio_pair(size_t buffer_size)
{
    BIO* int_bio = nullptr;
    BIO* ext_bio = nullptr;
    BIO_new_bio_pair(&ext_bio, buffer_size, &int_bio, buffer_size);
    return { ext_bio, int_bio };
}

inline bool bio_has_data(BIO* bio)
{
    return ::BIO_ctrl_wpending(bio);
}

inline void free_bio_pair(BIO* ext_bio, BIO* int_bio)
{
    ::BIO_free(ext_bio);
    ::BIO_free(int_bio);
}

inline void replace_bio_pair(
    BIO** ext_bio,
    BIO* ext_bio_new,
    BIO* int_bio_new,
    ssl_stream_core& core)
{
    auto ssl = core.engine_.native_handle();
    ::BIO_free(*ext_bio);
    *ext_bio = ext_bio_new;
    SSL_set_bio(ssl, int_bio_new, int_bio_new);
}

template <typename Socket>
inline void reduce_engine_buffers(ssl_stream<Socket>& stream, size_t buffer_size)
{
    auto core = get_core(stream);
    if (!core) return;
    auto [ext_bio, int_bio] = extract_bio_pair(*core);
    auto [ext_bio_new, int_bio_new] = make_new_bio_pair(buffer_size);
    if (bio_has_data(*ext_bio))
    {
        try
        {
            // Reuse output_buffer to prevent extra allocation.
            copy_data_from_bio(int_bio, ext_bio_new, core->output_buffer_);
        }
        catch (const std::exception&)
        {
            free_bio_pair(ext_bio_new, int_bio_new);
            throw;
        }
    }
    replace_bio_pair(ext_bio, ext_bio_new, int_bio_new, *core);
}

/*
    ssl::stream contains 4 buffers of 17KB each: input/output buffers
    inside stream_core and 2 buffers associated with ssl::detail::engine.

    OpenSSL itself has buffers and we can resize ssl::stream buffers to reduce
    memory consumption.

    Do not call this function while there are active operations on ssl::stream.
*/
template <typename Socket>
inline void reduce_ssl_buffers(ssl_stream<Socket>& stream, size_t buffer_size)
{
    reduce_engine_buffers(stream, buffer_size);
    reduce_stream_core_buffers(stream, buffer_size);
}

}}}}

namespace yplatform {

template struct private_access<
    net::stream::detail::ssl_stream_core_tag,
    &net::stream::detail::ssl_tcp_stream::core_>;

template struct private_access<
    net::stream::detail::ext_bio_tag,
    &boost::asio::ssl::detail::engine::ext_bio_>;

}
