#pragma once

#include <openssl/ssl.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string>

namespace yplatform { namespace net { namespace stream { namespace detail {

inline void put_hex(std::string& buffer, char c)
{
    int c1 = static_cast<unsigned char>(c) >> 4;
    int c2 = c & 0xF;
    buffer += static_cast<char>(c1 < 10 ? '0' + c1 : 'A' + c1 - 10);
    buffer += static_cast<char>(c2 < 10 ? '0' + c2 : 'A' + c2 - 10);
}

inline std::string dump(
    unsigned char* client_random,
    unsigned char* master_key,
    int master_key_length)
{
    static const std::string PREFIX = "CLIENT_RANDOM ";
    std::string line;
    line.reserve(PREFIX.size() + 2 * SSL3_RANDOM_SIZE + 1 + 2 * SSL_MAX_MASTER_KEY_LENGTH + 1);

    line += PREFIX;
    for (int i = 0; i < SSL3_RANDOM_SIZE; i++)
    {
        put_hex(line, client_random[i]);
    }
    line += ' ';
    for (int i = 0; i < master_key_length; i++)
    {
        put_hex(line, master_key[i]);
    }
    line += '\n';
    return line;
}

struct ssl_tap_state
{
    int master_key_length;
    unsigned char master_key[SSL_MAX_MASTER_KEY_LENGTH];
};

inline SSL_SESSION* ssl_get_session(const SSL* ssl)
{
#if OPENSSL_VERSION_NUMBER >= 0x10100000L
    return SSL_get_session(ssl);
#else
    return ssl->session;
#endif
}

inline void copy_master_secret(
    const SSL_SESSION* session,
    unsigned char* master_key_out,
    int* keylen_out)
{
#if OPENSSL_VERSION_NUMBER >= 0x10100000L
    *keylen_out = SSL_SESSION_get_master_key(session, master_key_out, SSL_MAX_MASTER_KEY_LENGTH);
#else
    if (session->master_key_length > 0 && session->master_key_length <= SSL_MAX_MASTER_KEY_LENGTH)
    {
        *keylen_out = session->master_key_length;
        memcpy(master_key_out, session->master_key, session->master_key_length);
    }
#endif
}

inline void copy_client_random(const SSL* ssl, unsigned char* client_random)
{
#if OPENSSL_VERSION_NUMBER >= 0x10100000L
    // ssl->s3 is not checked in openssl 1.1.0-pre6, but let's assume that
    // we have a valid SSL context if we have a non-NULL session.
    SSL_get_client_random(ssl, client_random, SSL3_RANDOM_SIZE);
#else
    if (ssl->s3)
    {
        memcpy(client_random, ssl->s3->client_random, SSL3_RANDOM_SIZE);
    }
#endif
}

// Copies SSL state for later comparison in tap_ssl_key.
inline void ssl_tap_state_init(const SSL* ssl, ssl_tap_state& state)
{
    const SSL_SESSION* session = ssl_get_session(ssl);
    memset(&state, 0, sizeof(ssl_tap_state));
    if (session)
    {
        copy_master_secret(session, state.master_key, &state.master_key_length);
    }
}

inline std::string ssl_tap_key(const SSL* ssl, const ssl_tap_state& state)
{
    const SSL_SESSION* session = ssl_get_session(ssl);
    unsigned char client_random[SSL3_RANDOM_SIZE];
    unsigned char master_key[SSL_MAX_MASTER_KEY_LENGTH];
    int master_key_length = 0;

    if (session)
    {
        copy_master_secret(session, master_key, &master_key_length);
        // Assume we have a client random if the master key is set.
        if (master_key_length > 0)
        {
            copy_client_random(ssl, client_random);
        }
    }

    if (master_key_length > 0)
    {
        // Skip writing keys if it did not change.
        if (state.master_key_length == master_key_length &&
            memcmp(state.master_key, master_key, master_key_length) == 0)
        {
            return std::string();
        }
        return dump(client_random, master_key, master_key_length);
    }

    return std::string();
}

}}}}
