#pragma once

#include <yplatform/net/buffer_sequence.h>
#include <yplatform/net/streambuf.h>
#include <yplatform/net/buffers/chunk.h>
#include <boost/shared_ptr.hpp>
#include <boost/asio/streambuf.hpp>

namespace yplatform { namespace net {

struct tag_streamer_base_no_log
{
};

class streamer_base
{
public:
    virtual ~streamer_base()
    {
        delete ostr;
        delete buf;
    }

    template <typename T>
    streamer_base& operator<<(T const& t)
    {
        *ostr << t;
        return *this;
    }

    streamer_base& operator<<(tag_streamer_base_no_log const& /*t*/)
    {
        skip_log = true;
        return *this;
    }

    streamer_base& write(const char* buffer, std::streamsize size)
    {
        ostr->write(buffer, size);
        return *this;
    }

protected:
    streamer_base()
        : buf(new yplatform::net::basic_streambuf<>), ostr(new std::ostream(buf)), skip_log(false)
    {
    }

    yplatform::net::basic_streambuf<>* buf;
    std::ostream* ostr;
    bool skip_log;
};

class streamer_wrapper
{
public:
    struct ref
    {
        streamer_base* p;
        ref(streamer_base* rhs) : p(rhs)
        {
        }
    };

    streamer_wrapper(streamer_wrapper&& wrp) : base_(std::move(wrp.base_))
    {
    }

    streamer_wrapper(streamer_base* base) : base_(base)
    {
    }

    streamer_wrapper& operator=(streamer_wrapper&& wrp)
    {
        base_ = std::move(wrp.base_);
        return *this;
    }

    streamer_wrapper(ref rhs) : base_(rhs.p)
    {
    }

    streamer_wrapper& operator=(ref rhs)
    {
        base_.reset(rhs.p);
        return *this;
    }

    operator ref()
    {
        return ref(base_.release());
    }

    template <typename T>
    inline streamer_wrapper& operator<<(T const& t)
    {
        *base_ << t;
        return *this;
    }

    streamer_wrapper& write(const char* buffer, std::streamsize size)
    {
        base_->write(buffer, size);
        return *this;
    }

private:
    std::unique_ptr<streamer_base> base_;
};

template <class Session>
class streamer : public streamer_base
{
public:
    streamer(boost::shared_ptr<Session> const& s) : sess(s)
    {
    }

    ~streamer()
    {
        ostr->flush();
        sess->send_client_stream2(buf->release_buffer(), skip_log);
    }

private:
    boost::shared_ptr<Session> sess;
};

struct streamable
{
    virtual ~streamable()
    {
    }

    void send(boost::asio::const_buffer const& s)
    {
        this->send_client_stream(buffers::make_const_asio_chunk(s));
    }

    void send(buffers::const_base_chunk* s)
    {
        this->send_client_stream(buffers::make_chunk_buffer(s));
    }

    void send(buffers::const_base_chunk_ptr s)
    {
        this->send_client_stream(buffers::make_chunk_buffer(s));
    }

    void send_string(string& s)
    {
        this->send_client_stream(buffers::make_const_stdcont_chunk(s));
    }

    void send_const_string(const string& s)
    {
        this->send_client_stream(buffers::make_const_cstdcont_chunk(s));
    }

    virtual void send_client_stream(buffers::const_chunk_buffer const& s) = 0;
    virtual void send_client_stream2(buffers::const_chunk_buffer const& s, bool /*skip_log*/)
    {
        send_client_stream(s);
    }

    virtual streamer_wrapper client_stream() = 0;
    virtual bool is_open() const = 0;
};

typedef boost::shared_ptr<streamable> streamable_ptr;

}}
