#pragma once

#include <yplatform/util/string_translation.h>
#include <yplatform/api.h>
#include <yplatform/time_traits.h>
#include <boost/optional.hpp>
#include <boost/property_tree/ptree.hpp>

namespace yplatform {
namespace detail {

template <typename T, typename = void>
struct is_container : std::false_type
{
};

template <typename T>
struct is_container<
    T,
    std::void_t<
        decltype(std::declval<T>().begin()),
        decltype(std::declval<T>().end()),
        typename T::value_type>> : std::true_type
{
};

template <typename T>
inline constexpr bool is_container_v = is_container<T>::value;

template <typename T, typename = void>
struct is_map : std::false_type
{
};

template <typename T>
struct is_map<T, std::enable_if_t<is_container_v<T>, std::void_t<typename T::mapped_type>>>
    : std::true_type
{
};

template <typename T>
inline constexpr bool is_map_v = is_map<T>::value;

}

using boost::property_tree::ptree;
typedef boost::optional<ptree> optional_ptree;

namespace detail {
struct BoolTranslator
{
    using internal_type = std::string;
    using external_type = bool;

    boost::optional<external_type> get_value(const internal_type& str)
    {
        return bool_from_string(str);
    }

    boost::optional<internal_type> put_value(const external_type& b)
    {
        return bool_to_string(b);
    }
};

struct DurationTranslator
{
    using internal_type = std::string;
    using external_type = time_traits::duration;

    boost::optional<external_type> get_value(const internal_type& str)
    {
        return duration_from_string(str);
    }

    boost::optional<internal_type> put_value(const external_type& b)
    {
        return duration_to_string(b);
    }
};
}

}

namespace boost { namespace property_tree {

template <>
struct translator_between<std::string, yplatform::time_traits::duration>
{
    typedef yplatform::detail::DurationTranslator type;
};

template <>
struct translator_between<std::string, bool>
{
    typedef yplatform::detail::BoolTranslator type;
};

}}

namespace yplatform {

template <class Type>
inline void ptree_convert(const ptree& conf, Type& out)
{
    try
    {
        out = conf.get_value<Type>();
    }
    catch (const boost::property_tree::ptree_error& ex)
    {
        throw std::runtime_error(
            std::string("can't cast value: \"") + conf.get_value<std::string>());
    }
}

template <template <class...> class Collection, class Type, class... Other>
inline void read_ptree_to_collection(
    Collection<Type, Other...>& collection,
    const ptree& node,
    const std::string& key)
{
    auto item = std::inserter(collection, collection.begin());
    auto range = node.equal_range(key);
    for (auto pt = range.first; pt != range.second; pt++)
    {
        Type value;
        ptree_convert(pt->second, value);
        item = std::move(value);
    }
}

template <typename Map>
inline void read_ptree_to_map(Map& map, const ptree& node, const std::string& key)
{
    using map_key = typename Map::key_type;
    using map_val = typename Map::mapped_type;
    for (auto&& [name, val] : node.get_child(key))
    {
        map_key res_key;
        ptree_convert(ptree(name), res_key);
        map_val res_val;
        ptree_convert(val, res_val);
        map.insert({ std::move(res_key), std::move(res_val) });
    }
}

template <typename T>
inline void read_ptree(T& out, const ptree& node, const std::string& key)
{
    if constexpr (detail::is_map_v<T>)
    {
        read_ptree_to_map(out, node, key);
    }
    else if constexpr (detail::is_container_v<T>)
    {
        read_ptree_to_collection(out, node, key);
    }
    else
    {
        ptree_convert(node.get_child(key), out);
    }
}

template <typename T>
inline void read_ptree(T& out, boost::optional<const ptree&> node, const std::string& key)
{
    if (node)
    {
        read_ptree(out, *node, key);
    }
}

}
