#pragma once

#include <yplatform/reactor/io_pool.h>
#include <yplatform/util/shared_ptr_cast.h>
#include <string>
#include <vector>
#include <map>

namespace yplatform {

struct reactor_cfg
{
    reactor_cfg() : io_threads(0), pool_count(0)
    {
    }

    reactor_cfg(unsigned threads, unsigned pools) : io_threads(threads), pool_count(pools)
    {
    }

    unsigned io_threads;
    unsigned pool_count;

    // Supported configurations:
    // 1. N x1 thread
    // reactor:
    //     size: N
    // 2. N xM threads
    // reactor:
    //     threads: M
    //     pool_count: M
    // 3. <reactor io_threads="M" pool_count="N">...</reactor>
    // 4. deprecated 1 xM threads
    // <io_threads>10</io_threads>
    void load(const ptree& cfg)
    {
        pool_count = cfg.get("size", 0);
        if (pool_count)
        {
            io_threads = 1;
        }
        else
        {
            io_threads = cfg.get("threads", 0);
            if (!io_threads) io_threads = cfg.get("<xmlattr>.io_threads", 0);
            pool_count = cfg.get("pool_count", 0);
            if (!pool_count) pool_count = cfg.get("<xmlattr>.pool_count", 0);
            if (!io_threads)
            {
                pool_count = 10;
                io_threads = cfg.get("io_threads", 1);
            }
            else if (!pool_count)
            {
                pool_count = 1;
            }
        }
    }
};

struct boost_thread_id_hash
{
    std::size_t operator()(boost::thread::id value) const
    {
        static_assert(
            sizeof(boost::thread::id) == sizeof(std::size_t),
            "sizeof(boost::thread::id) != sizeof(std::size_t)");
        return *reinterpret_cast<std::size_t*>(&value);
    }
};

class reactor
{
    typedef boost::asio::io_service::work work;
    typedef boost::shared_ptr<boost::asio::io_service::work> work_ptr;

public:
    reactor();

    reactor(std::shared_ptr<io_pool> external_pool);
    reactor(boost::asio::io_service& io, size_t concurrency_hint);

    void init(unsigned pool_count, unsigned thread_count);

    void init(const reactor_cfg& cfg);

    void run();

    void stop();

    void fini();

    const std::string& name() const;

    void set_name(const std::string& name);

    void set_logger(const yplatform::log::source& lg);

    std::size_t get_index() const;

    const io_pool_ptr& get_pool() const;

    const std::vector<io_pool_ptr>& get_pools() const;

    boost::asio::io_service* io();

    const boost::asio::io_service* io() const;

    const io_pool_ptr& operator[](std::size_t i) const;

    size_t size() const;

    // TODO update other projects
    // All io pools run only 1 thread.
    bool plain() const;

    static boost::shared_ptr<reactor> make_not_owning_copy(reactor& external_reactor);

    template <typename Handler>
    void post(Handler&& handler)
    {
        io()->post(std::forward<Handler>(handler));
    }

private:
    void init();

    mutable std::atomic_size_t next_io_data_;
    std::atomic_bool stopped_;
    std::vector<io_pool_ptr> pools_;
    std::vector<work_ptr> work_;
    reactor_cfg cfg_;
    std::string name_;
    std::unordered_map<boost::thread::id, std::size_t, boost_thread_id_hash> threads_pools_map_;
};

typedef boost::shared_ptr<reactor> reactor_ptr;

struct reactor_set
{
public:
    reactor_ptr add(const std::string& name, const reactor_cfg& cfg);

    reactor_ptr add(const std::string& name, const reactor_ptr& reactor);

    reactor_ptr get(const std::string& name);

    bool exists(const std::string& name) const;

    reactor_ptr get_global();

    void run();

    void stop();

    void fini();

    const std::map<std::string, reactor_ptr>& get_all();

private:
    std::map<std::string, reactor_ptr> reactors_;
    reactor_ptr global_;
};

typedef boost::shared_ptr<reactor_set> reactor_set_ptr;

extern API_PUBLIC reactor_set_ptr global_reactor_set;
extern API_PUBLIC reactor_ptr global_net_reactor;

}
