#pragma once

#include <yplatform/time_traits.h>
#include <yplatform/task_context.h>
#include <boost/asio/io_service.hpp>
#include <memory>

namespace yplatform {

class reactor_tracer : public std::enable_shared_from_this<reactor_tracer>
{
public:
    reactor_tracer(boost::asio::io_service& io) : io_(io), timer_(io)
    {
    }

    void start()
    {
        trace();
    }

    const time_traits::duration& last_delay()
    {
        return last_delay_;
    }

private:
    void trace(boost::system::error_code err = {})
    {
        if (err)
        {
            return;
        }
        trace_begin_ts_ = time_traits::clock::now();
        io_.post(std::bind(&reactor_tracer::handle_trace, shared_from_this()));
    }

    void wait_and_trace()
    {
        timer_.expires_from_now(frequency_);
        timer_.async_wait(
            std::bind(&reactor_tracer::trace, shared_from_this(), std::placeholders::_1));
    }

    void handle_trace()
    {
        last_delay_ = time_traits::clock::now() - trace_begin_ts_;
        if (last_delay_ >= frequency_)
        {
            trace();
        }
        else
        {
            wait_and_trace();
        }
    }

    static constexpr time_traits::duration frequency_ = time_traits::milliseconds(1);

    boost::asio::io_service& io_;
    time_traits::timer timer_;
    time_traits::time_point trace_begin_ts_;
    time_traits::duration last_delay_ = time_traits::milliseconds(0);
};

using reactor_tracer_ptr = std::shared_ptr<reactor_tracer>;

struct trace_service : boost::asio::io_service::service
{
    static boost::asio::io_service::id id;

    trace_service(boost::asio::io_service& io) : boost::asio::io_service::service(io)
    {
    }

    reactor_tracer_ptr tracer()
    {
        return tracer_;
    }

    void tracer(reactor_tracer_ptr tracer)
    {
        tracer_ = tracer;
    }

    reactor_tracer_ptr tracer_;
};

inline reactor_tracer_ptr find_tracer(boost::asio::io_service& io)
{
    auto& service = boost::asio::use_service<trace_service>(io);
    return service.tracer();
}

inline void store_tracer(boost::asio::io_service& io, reactor_tracer_ptr tracer)
{
    auto& service = boost::asio::use_service<trace_service>(io);
    service.tracer(tracer);
}

}
