#pragma once

#include <boost/function.hpp>
#include <boost/thread/locks.hpp>
#include <boost/thread/mutex.hpp>
#include <list>

#define DEBUG_CALLBACK_LIST 0

namespace yplatform {

template <typename Obj>
class callback_list
{
public:
    typedef typename std::list<Obj>::iterator iterator;
    typedef typename std::list<Obj>::const_iterator const_iterator;
    typedef typename std::list<Obj>::size_type size_type;
    typedef boost::function<void(void)> callback_t;

    callback_list() : closed_(false), end_i_(list_.end()), call_size_(1), count_(0), all_size_(0)
    {
    }

    callback_list(size_type call_size)
        : closed_(false), end_i_(list_.end()), call_size_(call_size), count_(0), all_size_(0)
    {
    }

    iterator begin()
    {
        return list_.begin();
    }

    const_iterator begin() const
    {
        return list_.begin();
    }

    iterator end()
    {
#if DEBUG_CALLBACK_LIST
        boost::unique_lock lock(mux_);
#endif
        return end_i_;
    }

    const_iterator end() const
    {
#if DEBUG_CALLBACK_LIST
        boost::unique_lock lock(mux_);
#endif
        return end_i_;
    }

    size_type size() const
    {
#if DEBUG_CALLBACK_LIST
        boost::unique_lock lock(mux_);
#endif
        return all_size_;
    }

    bool empty() const
    {
        return begin() == end();
    }

    bool is_closed() const
    {
#if DEBUG_CALLBACK_LIST
        boost::unique_lock lock(mux_);
#endif
        return closed_;
    }

    void push(const Obj& obj)
    {
#if DEBUG_CALLBACK_LIST
        {
            boost::unique_lock lock(mux_);
            BOOST_ASSERT(!closed_);
        }
#endif
        list_.push_back(obj);
        count_++;
        closed_ = last;

        if (count_ < call_size_) return;
        boost::unique_lock lock(mux_);
        call_callback();
    }

    void close()
    {
        boost::unique_lock lock(mux_);
#if DEBUG_CALLBACK_LIST
        BOOST_ASSERT(!closed_);
#endif
        closed_ = true;
        call_callback();
    }

    void set_callback(callback_t callback)
    {
        boost::unique_lock lock(mux_);
        callback_ = callback;
        if (closed_) call_callback();
    }

protected:
    void call_callback()
    {
        if (!callback_) return;
        callback_t cb = callback_;
        callback_ = callback_t();
        all_size_ += count_;
        end_i_ = list_.end();
        count_ = 0;
        cb();
    }

private:
    std::list<Obj> list_;
    bool closed_;
    iterator end_i_;
    callback_t callback_;
    size_type call_size_;
    size_type count_;
    size_type all_size_;
    boost::mutex mux_;
};

}
