#pragma once

#include <algorithm>
#include <ctime>
#include <stdexcept>
#include <vector>

namespace yplatform {

template <typename T>
class per_second_accumulator
{
public:
    typedef T value_type;

    per_second_accumulator(size_t max_size) : values_(max_size, value_type())
    {
        if (max_size == 1) throw std::runtime_error("invalid size of per_second_accumulator");

        last_modified_ = 0;
    }

    value_type add(value_type t)
    {
        return add(std::time(0), t);
    }

    value_type add(std::time_t now, value_type t)
    {
        check_previous(now);

        size_t pos = position(now);
        values_[pos] += t;
        return values_[pos];
    }

    void update(value_type t)
    {
        update(std::time(0), t);
    }

    void update(std::time_t now, value_type t)
    {
        check_previous(now);

        size_t pos = position(now);
        values_[pos] = t;
    }

    // get value for previous full second
    value_type get_last()
    {
        return get_last(std::time(0));
    }

    value_type get_last(std::time_t now)
    {
        return get_current(now - 1);
    }

    // get value for current second
    value_type get_current()
    {
        return get_current(std::time(0));
    }

    value_type get_current(std::time_t now)
    {
        check_previous(now);

        size_t pos = position(now);
        return values_[pos];
    }

    value_type get_avg(unsigned last_count)
    {
        return get_avg(std::time(0), last_count);
    }

    // not pre-calculated, no need for now
    value_type get_avg(std::time_t now, unsigned last_count)
    {
        if (last_count == 0) return value_type();

        return get_sum(now, last_count) / last_count;
    }

    value_type get_sum()
    {
        return get_sum(std::time(0), values_.size() - 1);
    }

    value_type get_sum(unsigned last_count)
    {
        return get_sum(std::time(0), last_count);
    }

    value_type get_sum(std::time_t now, unsigned last_count)
    {
        if (last_count == 0) return value_type();

        check_previous(now - 1);

        size_t corrected_count = std::min<size_t>(last_count, values_.size() - 1);
        size_t pos_from = position(now - corrected_count);
        size_t pos_to = position(now - 1);

        if (pos_from == pos_to) return values_[pos_from];

        value_type result = value_type();
        for (size_t i = 0; i < corrected_count; ++i)
        {
            result += values_[round_position(pos_from + i)];
        }
        return result;
    }

    value_type get_max()
    {
        return get_max(std::time(0), values_.size() - 1);
    }

    value_type get_max(unsigned last_count)
    {
        return get_max(std::time(0), last_count);
    }

    value_type get_max(std::time_t now, unsigned last_count)
    {
        if (last_count == 0) return value_type(0);

        check_previous(now - 1);

        size_t corrected_count = std::min<size_t>(last_count, values_.size() - 1);
        size_t pos_from = position(now - corrected_count);
        size_t pos_to = position(now - 1);

        if (pos_from == pos_to) return values_[pos_from];

        value_type result = value_type(0);
        for (size_t i = 0; i < corrected_count; ++i)
        {
            auto& value = values_[round_position(pos_from + i)];
            if (value > result)
            {
                result = value;
            }
        }
        return result;
    }

    size_t capacity() const
    {
        return values_.size();
    }

private:
    size_t position(std::time_t time)
    {
        return static_cast<size_t>(time % values_.size());
    }

    size_t round_position(size_t pos)
    {
        return pos % values_.size();
    }

    void check_previous(std::time_t time)
    {
        if (last_modified_ < time)
        {
            int distance = static_cast<int>(time - last_modified_);
            distance = std::min<int>(values_.size(), distance);
            for (int i = 1; i <= distance; ++i)
            {
                values_[position(last_modified_ + i)] = value_type();
            }
            last_modified_ = time;
        }
    }
    std::vector<value_type> values_;
    std::time_t last_modified_;
};

}