#pragma once

#include <yplatform/log.h>
#include <yplatform/application/task_context.h>
#include <string>
#include <type_traits>

namespace yplatform {
namespace detail {
template <typename CharType>
inline bool is_empty(const CharType* msg)
{
    return !std::char_traits<CharType>::length(msg);
}

template <typename... Args>
inline bool is_empty(const std::basic_string<Args...>& msg)
{
    return msg.empty();
}

template <typename T>
inline bool is_empty(const T&)
{
    return false;
}

template <typename T>
auto make_empty_result()
{
    if constexpr (!std::is_void_v<T>) return std::optional<T>();
}

template <typename T>
struct safe_result
{
    using type = std::optional<T>;
};

template <>
struct safe_result<void>
{
    using type = void;
};

template <typename T>
using safe_result_t = typename safe_result<T>::type;

}

template <
    typename MessageType,
    typename Handler,
    typename... Args,
    typename Result = std::invoke_result_t<Handler, Args...>>
inline detail::safe_result_t<Result> safe_call(
    MessageType&& what,
    Handler&& handler,
    Args&&... args)
{
    try
    {
        return handler(std::forward<Args>(args)...);
    }
    catch (const std::exception& ex)
    {
        YLOG_G(error) << what << (detail::is_empty(what) ? "" : " ")
                      << "handler caused an error: \"" << ex.what() << "\"";
        return detail::make_empty_result<Result>();
    }
    catch (...)
    {
        YLOG_G(error) << what << (detail::is_empty(what) ? "" : " ") << "handler caused an error";
        return detail::make_empty_result<Result>();
    }
}

template <
    typename MessageType,
    typename Handler,
    typename... Args,
    typename Result = std::invoke_result_t<Handler, Args...>>
inline detail::safe_result_t<Result> safe_call(
    const task_context_ptr& ctx,
    MessageType&& what,
    Handler&& handler,
    Args&&... args)
{
    try
    {
        return handler(std::forward<Args>(args)...);
    }
    catch (const std::exception& ex)
    {
        YLOG_CTX_GLOBAL(ctx, error) << what << (detail::is_empty(what) ? "" : " ")
                                    << "handler caused an error: \"" << ex.what() << "\"";
        return detail::make_empty_result<Result>();
    }
    catch (...)
    {
        YLOG_CTX_GLOBAL(ctx, error)
            << what << (detail::is_empty(what) ? "" : " ") << "handler caused an error";
        return detail::make_empty_result<Result>();
    }
}

template <typename Handler, typename... Args, typename = std::invoke_result_t<Handler, Args...>>
inline auto safe_call(Handler&& handler, Args&&... args)
{
    return safe_call("", std::forward<Handler>(handler), std::forward<Args>(args)...);
}

template <typename Handler, typename... Args, typename = std::invoke_result_t<Handler, Args...>>
inline auto safe_call(const task_context_ptr& ctx, Handler&& handler, Args&&... args)
{
    return safe_call(ctx, "", std::forward<Handler>(handler), std::forward<Args>(args)...);
}

}
