#pragma once

#include <yplatform/util/split.h>
#include <yplatform/time_traits.h>
#include <boost/optional.hpp>
#include <numeric>
#include <string>

namespace yplatform {

inline boost::optional<bool> bool_from_string(const std::string& str)
{
    if (!str.empty())
    {
        if (str == "yes" || str == "on" || str == "1" || str == "true")
        {
            return true;
        }
        else if (str == "no" || str == "off" || str == "0" || str == "false")
        {
            return false;
        }
        else
        {
            throw std::domain_error("invalid bool value");
        }
    }
    else
    {
        return {};
    }
}

inline std::string bool_to_string(bool val)
{
    return val ? "true" : "false";
}

inline boost::optional<time_traits::duration> duration_from_string(const std::string& str)
{
    static const std::map<std::string, std::function<time_traits::duration(float)>> converters{
        { "h", [](float val) { return time_traits::minutes(static_cast<unsigned>(val * 60)); } },
        { "m", [](float val) { return time_traits::seconds(static_cast<unsigned>(val * 60)); } },
        { "s",
          [](float val) { return time_traits::milliseconds(static_cast<unsigned>(val * 1000)); } },
        { "ms",
          [](float val) { return time_traits::microseconds(static_cast<unsigned>(val * 1000)); } },
        { "us",
          [](float val) { return time_traits::nanoseconds(static_cast<unsigned>(val * 1000)); } }
    };

    if (str.empty()) return {};
    else if (auto parts = util::split(str, ":"); parts.size() == 3)
    {
        auto sec = std::accumulate(parts.begin(), parts.end(), 0ul, [](auto sec, auto& cur) {
            return sec * 60 + std::stoul(cur);
        });
        return converters.at("s")(static_cast<float>(sec));
    }
    else
    {
        size_t sz;
        auto val = std::stof(str, &sz);
        auto suffix = str.substr(sz);
        auto it = converters.find(suffix.empty() ? "s" : suffix);
        if (it == converters.end())
        {
            throw std::domain_error("unknown duration suffix");
        }
        return it->second(val);
    }
}

inline std::string duration_to_string(const time_traits::duration& b)
{
    auto float_b = time_traits::duration_cast<time_traits::float_seconds>(b);
    return std::to_string(float_b.count());
}

}
