#pragma once

#include <yplatform/time_traits.h>
#include <boost/uuid/uuid_io.hpp>
#include <boost/uuid/random_generator.hpp>
#include <boost/random/seed_seq.hpp>
#include <boost/random/mersenne_twister.hpp>
#include <mutex>
#include <string>

namespace yplatform { namespace util {

// default seeding function in boost::uuid uses
// uninitialized memory to generate entropy,
// which causes a lot of hard to suppress errors
// in valgrind. That's why it's preferrable to
// use another one. This one is based on current
// steady_clock time in nanoseconds, and should be
// "random" enough for most uses.
template <typename UniformRandomNumberGenerator, typename SeedSeq = boost::random::seed_seq>
void seed_with_now_nsec(UniformRandomNumberGenerator& gen)
{
    using namespace time_traits;
    auto now_nsec = duration_cast<nanoseconds>(clock::now().time_since_epoch()).count();

    constexpr std::size_t num_uints = sizeof(now_nsec) / 4;
    std::array<uint32_t, num_uints> init_sequence;
    for (std::size_t i = 0; i < num_uints; ++i)
    {
        init_sequence[i] = static_cast<uint32_t>(now_nsec);
        now_nsec >>= 32;
    }

    auto seed_sequence = SeedSeq(init_sequence.begin(), init_sequence.end());
    gen.seed(seed_sequence);
}

template <typename UniformRandomNumberGenerator>
class basic_string_uuid_generator
{
public:
    using random_number_generator_t = UniformRandomNumberGenerator;
    using seed_func_t = std::function<void(random_number_generator_t&)>;

    basic_string_uuid_generator(
        const seed_func_t& seed = seed_with_now_nsec<random_number_generator_t>)
        : gen_(number_gen_)
    {
        seed(number_gen_);
    }

    std::string operator()() const
    {
        std::lock_guard<std::mutex> lock(gen_mutex_);
        return boost::uuids::to_string(gen_());
    }

private:
    mutable std::mutex gen_mutex_;

    mutable random_number_generator_t number_gen_;
    mutable boost::uuids::basic_random_generator<random_number_generator_t> gen_;
};

using string_uuid_generator = basic_string_uuid_generator<boost::random::mt19937>;

}}
