#pragma once

#include <yplatform/zerocopy/fragment.h>

#include <boost/iterator/iterator_facade.hpp>

#include <algorithm>
#include <cassert>
#include <list>

namespace yplatform { namespace zerocopy {

template <typename Fragment, typename SegmentSeq>
class iterator_base
{
protected:
    typedef SegmentSeq fragment_list;
    typedef typename fragment_list::value_type::element_type fragment_type;
    typedef typename fragment_list::const_iterator fragment_iterator;
    typedef typename fragment_type::iterator skip_iterator;

    template <typename>
    friend class basic_segment;
    template <typename, typename, typename, typename>
    friend class basic_streambuf;

    fragment_list const* fragment_list_;

    fragment_iterator fragment_;
    skip_iterator pos_;
    bool after_last_frag_;

    iterator_base() : fragment_list_(nullptr), after_last_frag_(false)
    {
    }

    iterator_base(
        fragment_list const& fragments,
        skip_iterator const& pos,
        fragment_iterator fragment)
        : fragment_list_(&fragments)
        , fragment_(fragment)
        , pos_(pos)
        , after_last_frag_(frag_list().empty() || pos_at_end())
    {
    }

    fragment_list const& frag_list() const
    {
        assert(fragment_list_ != 0);
        return *fragment_list_;
    }

    bool pos_at_end() const
    {
        return pos_ == (*fragment_)->end();
    }
};

template <
    typename Value,
    typename Fragment = detail::fragment,
    typename SegmentSeq = std::list<boost::shared_ptr<Fragment>>>
class iterator
    : public boost::iterator_facade<
          iterator<Value, Fragment, SegmentSeq>,
          Value,
          boost::random_access_traversal_tag>
    , public iterator_base<Fragment, SegmentSeq>
{
    typedef boost::iterator_facade<
        iterator<Value, Fragment, SegmentSeq>,
        Value,
        boost::random_access_traversal_tag>
        iterator_facade_;

protected:
    typedef iterator_base<Fragment, SegmentSeq> base;

    typedef typename base::fragment_list fragment_list;
    typedef typename base::fragment_type fragment_type;
    typedef typename base::fragment_iterator fragment_iterator;
    typedef typename base::skip_iterator skip_iterator;

public:
    typedef typename iterator_facade_::difference_type difference_type;

    iterator() = default;

    iterator(const base& other) : base(other)
    {
    }

    iterator(fragment_list const& fragments, skip_iterator const& pos, fragment_iterator fragment)
        : base(fragments, pos, fragment)
    {
    }

private:
    friend class boost::iterator_core_access;

    bool equal_fragment_and_pos(iterator const& other) const
    {
        if (this->after_last_frag_ != other.after_last_frag_)
        {
            return fine_fragment() == other.fine_fragment() && fine_pos() == other.fine_pos();
        }
        return this->fragment_ == other.fragment_ && this->pos_ == other.pos_;
    }

    bool equal(iterator const& other) const
    {
        return this->fragment_list_ == other.fragment_list_ && equal_fragment_and_pos(other);
    }

    void increment()
    {
        assert(this->fragment_ != this->frag_list().end());
        assert(this->pos_ != (*this->fragment_)->end() || !is_last_fragment());

        if (this->after_last_frag_)
        {
            assert(!is_last_fragment());
            ++this->fragment_;
            this->pos_ = (*this->fragment_)->begin();
            this->after_last_frag_ = false;
        }
        ++this->pos_;
        if (this->pos_at_end())
        {
            if (is_last_fragment())
            {
                this->after_last_frag_ = true;
            }
            else
            {
                ++this->fragment_;
                this->pos_ = (*this->fragment_)->begin();
            }
        }
    }

    typename iterator_facade_::reference dereference() const
    {
        return *fine_pos();
    }

    void decrement()
    {
        if (this->after_last_frag_)
        {
            this->after_last_frag_ = false;
        }
        else if (this->pos_ == (*this->fragment_)->begin())
        {
            assert(this->fragment_ != this->frag_list().begin());
            --this->fragment_;
            this->pos_ = (*this->fragment_)->end();
        }
        --this->pos_;
    }

    void add_position(difference_type n)
    {
        assert(n >= 0);

        while (n > ((*this->fragment_)->end() - this->pos_))
        {
            n -= ((*this->fragment_)->end() - this->pos_);
            if (is_last_fragment())
            {
                this->pos_ = (*this->fragment_)->end();
                n = 0;
            }
            else
            {
                ++this->fragment_;
                assert(this->fragment_ != this->frag_list().end());
                this->pos_ = (*this->fragment_)->begin();
            }
        }

        assert(!(n != 0 && this->pos_ == (*this->fragment_)->end() && is_last_fragment()));
        this->pos_ += n;
    }

    void substract_position(difference_type n)
    {
        assert(n <= 0);

        while (n < ((*this->fragment_)->begin() - this->pos_))
        {
            assert(this->fragment_ != this->frag_list().begin());
            n -= ((*this->fragment_)->begin() - this->pos_);
            --this->fragment_;
            this->pos_ = (*this->fragment_)->end();
            --this->pos_;
            ++n;
        }

        this->pos_ += n;
    }

    void advance(difference_type n)
    {
        if (n > 0)
        {
            add_position(n);
        }
        else
        {
            substract_position(n);
        }
        this->after_last_frag_ = this->pos_at_end();
    }

    difference_type distance_to(iterator const& other) const
    {
        if (!this->fragment_list_ && !other.fragment_list_)
        {
            return 0;
        }

        assert(&this->frag_list() == &other.frag_list());

        difference_type result = 0;

        if (this->fragment_ == other.fragment_)
        {
            return other.pos_ - this->pos_;
        }

        if (!is_last_fragment())
        {
            result = (*this->fragment_)->end() - this->pos_;
            fragment_iterator f = this->fragment_;
            for (++f; f != other.fragment_ && f != this->frag_list().end(); ++f)
            {
                result += (*f)->size();
            }
            if (f == other.fragment_)
            {
                result += (other.pos_ - (*f)->begin());
                return result;
            }
        }

        assert(this->fragment_ != this->frag_list().begin());

        result = this->pos_ - (*this->fragment_)->begin();
        fragment_iterator f = this->fragment_;
        for (--f; f != other.fragment_ && f != this->frag_list().begin(); --f)
        {
            result += (*f)->size();
        }

        assert(f == other.fragment_);

        result += ((*f)->end() - other.pos_);

        return -result;
    }

    fragment_iterator fine_fragment() const
    {
        return this->after_last_frag_ && !is_last_fragment() ? next_fragment() : this->fragment_;
    }

    skip_iterator fine_pos() const
    {
        return this->after_last_frag_ && !is_last_fragment() ? next_pos() : this->pos_;
    }

    fragment_iterator next_fragment() const
    {
        assert(!is_last_fragment());
        auto i = this->fragment_;
        ++i;
        return i;
    }

    skip_iterator next_pos() const
    {
        return (*next_fragment())->begin();
    }

    bool is_last_fragment() const
    {
        assert(!this->frag_list().empty());
        auto i = this->frag_list().end();
        --i;
        return this->fragment_ == i;
    }

    template <typename>
    friend class basic_segment;
    template <typename, typename, typename, typename>
    friend class basic_streambuf;
};

} // namespace zerocopy
} // namespace yplatform
