#include <yplatform/application/application.h>
#include <yplatform/application/global_init.h>
#include <yplatform/application/context_repository.h>
#include <yplatform/application/app_service.h>
#include <yplatform/log.h>
#include <yplatform/reactor/reactor.h>
#include <sys/prctl.h>

namespace yplatform {

namespace {
void daemonize(void)
{
    pid_t pid = fork();
    if (pid < 0)
    {
        std::cerr << "daemonize: fork() failed, errno: " << errno;
        exit(EXIT_FAILURE);
    }

    if (pid > 0) exit(EXIT_SUCCESS);
}

bool detach_console()
{
    // Change the file mode mask
    umask(0);
#ifndef _PATH_DEVNULL
#error Please define _PATH_DEVNULL (/dev/null?)
#endif

    // prevent reuse of stdin/stdout/stderr by redirecting it to /dev/null
    int tmp_fd;

    if ((tmp_fd = open(_PATH_DEVNULL, O_RDONLY)) == -1)
    {
        log_fatal_error("cannot open DEVNULL for input");
        return false;
    }

    if (tmp_fd != STDIN_FILENO)
    {
        ::dup2(tmp_fd, STDIN_FILENO);
        ::close(tmp_fd);
    }

    if ((tmp_fd = open(_PATH_DEVNULL, O_WRONLY)) == -1)
    {
        log_fatal_error("cannot open DEVNULL for output");
        return false;
    }

    ::dup2(tmp_fd, STDOUT_FILENO);
    ::dup2(tmp_fd, STDERR_FILENO);

    if (tmp_fd != STDOUT_FILENO && tmp_fd != STDERR_FILENO) ::close(tmp_fd);

    return true;
}
}

application::application() : global_repository_(nullptr)
{
    log::init_global_log_console();
}

application::application(const configuration& config) : global_repository_(nullptr)
{
    try
    {
        log::init_global_log_console();
        if (config.daemon()) daemonize();
        load(config);
    }
    catch (std::exception const& e)
    {
        try
        {
            core_cleanup();
        }
        catch (...)
        {
        } // ignore exceptions, throw only load_error
        throw load_error("application::load status=error, reason: " + std::string(e.what()));
    }
    catch (...)
    {
        try
        {
            core_cleanup();
        }
        catch (...)
        {
        }
        throw load_error("application::load status=error, reason: unknown");
    }

    if (config.daemon() && !detach_console())
    {
        try
        {
            core_cleanup();
        }
        catch (...)
        {
        }
        throw load_error("failed to detach_console");
    }
}

application::~application()
{
    try
    {
        core_cleanup();
        log::global_log_deinit();
    }
    catch (std::exception const& e)
    {
        YLOG_GLOBAL(alert) << "application::~application status=error, reason: "
                           << std::string(e.what());
    }
    catch (...)
    {
        YLOG_GLOBAL(alert) << "application::~application status=error, reason: unknown";
    }
}

void application::load(const configuration& cfg)
{
    if (!cfg.dir().empty() && ::chdir(cfg.dir().c_str()))
    {
        throw std::runtime_error("Cannot chdir to " + cfg.dir());
    }

    if (cfg.gid() && ::setgid(cfg.gid()) == -1)
    {
        throw std::runtime_error("cannot change process gid");
    }

    // This assures that context_repository::instance() will be available
    // for entire lifetime of application object
    init_global_context_repository();

    log::global_log_init(cfg.log_cfg());

    if (cfg.environment().size())
    {
        YLOG_GLOBAL(info) << "starting in " << cfg.environment() << " environment";
    }

    reactors_ = boost::make_shared<reactor_set>();
    global_reactor_set = boost::make_shared<reactor_set>();
    for (auto& pair : cfg.io_cfg())
    {
        auto reactor = reactors_->add(pair.first, pair.second);
        auto copy = reactor::make_not_owning_copy(*reactor);
        global_reactor_set->add(pair.first, copy);
    }
    global_net_reactor = global_reactor_set->get_global();
    for (auto& pair : reactors_->get_all())
    {
        auto& reactor = pair.second;
        for (auto& pool : reactor->get_pools())
        {
            log::init(*pool->io(), log::detail::global_service::instance().get_ptr());
            store_module_repository(*pool->io(), repository::instance_ptr());
            store_context_repository(*pool->io(), context_repository::instance_ptr());
        }
    }

    global_repository_ = &repository::instance();

    // can throw - global reactors will be deleted on program terminate
    init_global_repository(global_reactor_set, cfg);

    if (!cfg.pid().empty()) pid_.reset(new pid_file(cfg.pid()));
    if (cfg.uid() && ::setuid(cfg.uid()) == -1)
    {
        throw std::runtime_error("cannot change process uid");
    }

    if ((cfg.gid() || cfg.uid()) && ::prctl(PR_SET_DUMPABLE, 1) == -1)
    {
        throw std::runtime_error("cannot set dumpable");
    }
}

void application::run()
{

    try
    {
        repository::instance().start();
        reactors_->run();
        YLOG_GLOBAL(alert) << "application::run status=ok";
    }
    catch (std::exception const& e)
    {
        throw runtime_error("application::run status=error, reason: " + std::string(e.what()));
    }
    catch (...)
    {
        throw runtime_error("application::run status=error, reason: unknown");
    }
}

void application::reload(const configuration& cfg)
{
    try
    {
        YGLOBAL_LOG_SERVICE.reopen_all();
        YGLOBAL_LOG_SERVICE.update_log_levels_from(cfg.log_cfg());
        repository::instance().reload(cfg);
        YLOG_GLOBAL(alert) << "application::reload status=ok";
    }
    catch (std::exception const& e)
    {
        YLOG_GLOBAL(alert) << "application::reload status=error, reason: " << e.what();
    }
    catch (...)
    {
        YLOG_GLOBAL(alert) << "application::reload status=error, reason: unknown exception";
    }
}

void application::stop(int signal)
{
    YLOG_GLOBAL(alert) << "application::stop called with signal=" << signal;
    try
    {
        core_stop();
        core_cleanup();
        YLOG_GLOBAL(alert) << "application::stop status=ok";
    }
    catch (std::exception const& e)
    {
        throw runtime_error("application::stop status=error, reason: " + std::string(e.what()));
    }
    catch (...)
    {
        throw runtime_error("application::stop status=error, reason: unknown");
    }
}

void application::core_stop()
{
    repository::instance().stop();

    // cancel all active tasks
    context_repository::table_ptr contexts = context_repository::instance().get_contexts();
    for (task_context_list::const_iterator i = contexts->begin(), i_end = contexts->end();
         i != i_end;
         ++i)
    {
        (*i)->cancel();
    }

    context_repository::instance().wait_for_finish();

    // stop reactors - just resets io_service work objects
    if (reactors_) reactors_->stop();
}

void application::core_cleanup()
{
    // fini repository
    repository::instance().fini();

    // fini reactors - will stop all io_service objects and join io_pool threads
    if (reactors_)
    {
        reactors_->fini();
    }
}

}
