#include <yplatform/application/configuration.h>
#include <yplatform/application/config/loader.h>
#include <boost/filesystem.hpp>
#include <boost/bind.hpp>

namespace yplatform {

configuration::configuration() : daemon_(false), verbose_(0), uid_(0), gid_(0)
{
    tab_["daemon"] = boost::bind(&configuration::parse_daemon, _1, _2);
    tab_["verbose"] = boost::bind(&configuration::parse_verbose, _1, _2);
    tab_["uid"] = boost::bind(&configuration::parse_uid, _1, _2);
    tab_["gid"] = boost::bind(&configuration::parse_gid, _1, _2);
    tab_["dir"] = boost::bind(&configuration::parse_dir, _1, _2);
    tab_["pid"] = boost::bind(&configuration::parse_pid, _1, _2);
    tab_["libpath"] = boost::bind(&configuration::parse_libpath, _1, _2);
}

void configuration::load_from_file(const string& path)
{
    ptree parsed_cfg;
    utils::config::loader::from_file(path, parsed_cfg);
    load_configuration(parsed_cfg);
    load_module_configuration(parsed_cfg);
}

void configuration::load_from_str(const string& source)
{
    ptree parsed_cfg;
    utils::config::loader::from_str(source, parsed_cfg);
    load_configuration(parsed_cfg);
    load_module_configuration(parsed_cfg);
}

void configuration::load_configuration(const ptree& cfg)
{
    const ptree& pt = cfg.get_child("config.system");
    for (const ptree::value_type& v : pt)
    {
        system_config_tab_t::const_iterator found = tab_.find(v.first);
        if (found != tab_.end()) found->second(*this, v.second.data());
    }

    environment_ = cfg.get<string>("environment", "");

    auto range = pt.equal_range("reactor");
    for (auto it = range.first; it != range.second; ++it)
    {
        std::string name = it->second.get("<xmlattr>.name", "global");
        if (name == "local") throw std::runtime_error("reactor name \"local\" is reserved");
        io_cfg_[name].load(it->second);
    }
    if (io_cfg_.count("global") == 0) io_cfg_["global"].load(pt);

    log_cfg_ = cfg.get_child("config.log");
}

void configuration::load_module_configuration(const ptree& cfg)
{
    const ptree& pt = cfg.get_child("config.modules");
    for (const ptree::value_type& v : pt)
    {
        if (v.first == "include")
        {
            auto file_path = include_path(v.second.get("<xmlattr>.file", ""));
            if (file_path.size())
            {
                ptree incl_conf;
                utils::config::loader::from_file(file_path, incl_conf);
                load_module_configuration(incl_conf);
            }
        }
        else if (v.first == "module")
        {
            module_data data;
            load_module_data(v.second, data);
            modules_.push_back(data);
        }
    }
}

void configuration::load_module_data(const ptree& pt, module_data& data)
{
    boost::optional<const ptree&> tree_part = pt.get_child_optional("system");
    if (tree_part) load_module_system_data(tree_part.get(), data);

    tree_part = pt.get_child_optional("configuration");
    if (tree_part) load_module_config_data(tree_part.get(), data);

    tree_part = pt.get_child_optional("log");
    if (tree_part) load_module_log_data(tree_part.get(), data);

    tree_part = pt.get_child_optional("include");
    if (tree_part)
    {
        for (ptree::const_iterator i = tree_part->begin(); i != tree_part->end(); ++i)
        {
            if (i->first != "file") continue;
            ptree incl_conf;
            auto file_path = include_path(i->second.get_value(""));
            if (file_path.size())
            {
                utils::config::loader::from_file(file_path, incl_conf);
            }
            load_module_data(incl_conf.get_child("config"), data);
        }
    }
}

void configuration::load_module_system_data(const ptree& pt, module_data& data)
{
    data.name = pt.get("name", "");
    data.log_id = pt.get("log_id", data.name);
    data.type = pt.get("type", "");
    data.factory = pt.get("factory", "");
    if (data.factory.empty())
    {
        data.factory = pt.get("class", "");
    }
}

void configuration::load_config_data(const ptree& source, ptree& target)
{
    for (auto& [key, value] : source)
    {
        if (key == "include")
        {
            auto file_path = include_path(value.get("<xmlattr>.file", ""));
            if (file_path.size())
            {
                ptree incl_conf;
                utils::config::loader::from_file(file_path, incl_conf);
                load_config_data(incl_conf.get_child("config"), target);
            }
        }
        else
        {
            load_config_data(
                value,
                target.add_child(yplatform::ptree::path_type(key, '\0'), ptree{ value.data() }));
        }
    }
}

void configuration::load_module_config_data(const ptree& pt, module_data& data)
{
    auto& options = data.options.count("options") ? data.options.get_child("options") :
                                                    data.options.put_child("options", ptree{});
    load_config_data(pt, options);
}

void configuration::load_module_log_data(const ptree& pt, module_data& data)
{
    const auto boost_node = pt.get_child_optional("boost");
    if (boost_node)
    {
        data.log_cfg.put_child("options", boost_node.get());
    }
    else
    {
        data.log_cfg.put_child("options", pt);
    }
}

string configuration::include_path(string path)
{
    using boost::filesystem::exists;
    // The configuration is processed before chdir so try a relative path too.
    return path.size() && (exists(path) || exists(path = dir_ + "/" + path)) ? path : "";
}

}
