#include <yplatform/application/context_repository.h>
#include <yplatform/api.h>
#include <boost/thread/locks.hpp>
#include <boost/thread/condition_variable.hpp>

namespace yplatform {

struct context_repository_impl
{
    virtual ~context_repository_impl()
    {
    }
    virtual API_PUBLIC void add_context(task_context_ptr context) = 0;
    virtual API_PUBLIC void rem_context(task_context_ptr context) = 0;
    virtual API_PUBLIC context_repository::table_ptr get_contexts() = 0;
    virtual API_PUBLIC void wait_for_finish() = 0;
};

class sharded_context_repository_impl : public context_repository_impl
{
    struct repo_shard
    {
        boost::mutex shard_mutex;
        boost::condition_variable shard_cond;
        task_context_list contexts_;

        void add_context(task_context_ptr context)
        {
            {
                boost::unique_lock<boost::mutex> lock(shard_mutex);
                contexts_.insert(context);
            }
            context->begin();
        }
        void rem_context(task_context_ptr context)
        {
            boost::unique_lock<boost::mutex> lock(shard_mutex);
            contexts_.erase(context);

            if (contexts_.empty()) shard_cond.notify_all();
        }

        void wait_for_finish()
        {
            boost::unique_lock<boost::mutex> lock(shard_mutex);
            shard_cond.wait(lock, [this] { return contexts_.empty(); });
        }
    };

    typedef boost::shared_ptr<repo_shard> repo_shard_ptr;

    enum
    {
        SHARD_FACTOR = 64
    };
    std::vector<repo_shard_ptr> shards;

public:
    sharded_context_repository_impl()
    {
        for (size_t i = 0; i < SHARD_FACTOR; i++)
            shards.push_back(repo_shard_ptr(new repo_shard()));
    }

    API_PUBLIC void add_context(task_context_ptr context) override
    {
        size_t index = get_shard_index(context->uniq_id());
        shards[index]->add_context(context);
    }

    API_PUBLIC void rem_context(task_context_ptr context) override
    {
        size_t index = get_shard_index(context->uniq_id());
        shards[index]->rem_context(context);
    }

    API_PUBLIC context_repository::table_ptr get_contexts() override
    {
        context_repository::table_ptr result(new task_context_list);
        for (auto shard : shards)
        {
            boost::unique_lock<boost::mutex> lock(shard->shard_mutex);
            result->insert(shard->contexts_.begin(), shard->contexts_.end());
        }
        return result;
    }
    API_PUBLIC void wait_for_finish() override
    {
        for (auto shard : shards)
        {
            shard->wait_for_finish();
        }
    }

protected:
    size_t get_shard_index(const string& context_id)
    {
        if (context_id.length() < 2) return 0;
        return (static_cast<size_t>(context_id[0]) + static_cast<size_t>(context_id[1])) %
            SHARD_FACTOR;
    }
};

//-----------------------------------------------------------------------------

context_repository::context_repository() : impl(new sharded_context_repository_impl())
{
}

API_PUBLIC void context_repository::add_context(task_context_ptr context)
{
    impl->add_context(context);
}

API_PUBLIC void context_repository::rem_context(task_context_ptr context)
{
    impl->rem_context(context);
}

context_repository::table_ptr context_repository::get_contexts()
{
    return impl->get_contexts();
}

API_PUBLIC void context_repository::wait_for_finish()
{
    impl->wait_for_finish();
}

API_PUBLIC std::shared_ptr<context_repository> context_repository::instance_ptr()
{
    static std::shared_ptr<context_repository> repo = std::make_shared<context_repository>();
    return repo;
}

API_PUBLIC context_repository& context_repository::instance()
{
    return *instance_ptr();
}

}
