#include <yplatform/loader.h>

#include <yplatform/application.h>
#include <boost/filesystem.hpp>
#include <cerrno>
#include <fstream>
#include <paths.h>
#include <pthread.h>
#include <resolv.h>
#include <signal.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

using namespace yplatform;

std::unique_ptr<configuration> read_config(const string& path)
{
    boost::filesystem::path conf_path(path);
    if (!boost::filesystem::exists(conf_path))
        throw std::runtime_error("сonfig file does not exist: " + path);
    if (boost::filesystem::is_directory(conf_path))
        throw std::runtime_error("сonfig path is a directory: " + path);

    std::unique_ptr<configuration> config(new configuration);
    config->load_from_file(path);
    return config;
}

int yplatform_start(const std::string& config_path)
{
    try
    {
        // Block all signals for background thread.
        sigset_t wait_mask;
        sigemptyset(&wait_mask);
        sigaddset(&wait_mask, SIGINT);
        sigaddset(&wait_mask, SIGTERM);
        sigaddset(&wait_mask, SIGHUP);
        sigaddset(&wait_mask, SIGPIPE);
        pthread_sigmask(SIG_BLOCK, &wait_mask, 0);

        application app(*read_config(config_path));
        app.run();

        // Wait for signal indicating time to shut down.
        int sig = 0;
        do
        {
            sigwait(&wait_mask, &sig);
            if (sig == SIGHUP) try
                {
                    app.reload(*read_config(config_path));
                    if (res_init())
                    {
                        YLOG_GLOBAL(error)
                            << "res_init failed errno=" << errno << " strerror=" << strerror(errno);
                    }
                }
                catch (std::exception const& e)
                {
                    YLOG_GLOBAL(alert) << "HUP signal process error, reason: " << e.what();
                    break;
                }
        } while (sig != SIGINT && sig != SIGQUIT && sig != SIGTERM);

        app.stop(sig);

        return 0;
    }
    catch (const load_error& e)
    {
        log_fatal_error(e.what());
        ::exit(100);
    }
    catch (const runtime_error& e)
    {
        log_fatal_error(e.what());
        abort();
    }
    catch (const std::exception& e)
    {
        log_fatal_error(e.what());
    }
    catch (...)
    {
        log_fatal_error("unknown");
    }
    return 1;
}
