#include <yplatform/application/repository.h>
#include <yplatform/api.h>
#include <yplatform/log.h>

#include <boost/bind.hpp>
#include <boost/filesystem/path.hpp>

#include <stdexcept>
#include <iostream>
#include <string>
#include <sstream>

#include <algorithm>
#include <unistd.h>
#include <mutex>

namespace yplatform {

namespace pt = boost::property_tree;

global_factory::module_wrapper global_factory::create(
    reactor_set_ptr reactors,
    const configuration::module_data& data)
{
    auto found = instance().factories_.find(data.factory);
    if (found != instance().factories_.end())
    {
        return found->second.create(reactors, data);
    }
    else
    {
        throw std::runtime_error("no such factory '" + data.factory + "'");
    }
}

global_factory& global_factory::instance()
{
    static global_factory impl;
    return impl;
}

void repository::load_module(reactor_set_ptr reactors, const configuration::module_data& data)
{
    YLOG_GLOBAL(info) << "module: name=" << data.name << ", " << data.type << ", " << data.factory
                      << "()";

    if (!add_service(reactors, data))
    {
        boost::throw_exception(std::runtime_error("can't load module " + data.name));
    }
}

bool repository::add_service(reactor_set_ptr reactors, const configuration::module_data& data)
{
    yplatform::log::log_load_cfg(data.log_cfg);
    auto wrapper = global_factory::create(reactors, data);
    insert_i(data.name, wrapper);
    return true;
}

API_PUBLIC void repository::init(reactor_set_ptr reactors, const configuration& cfg)
{
    dir_ = cfg.dir();
    for (configuration::lib_path_lst::const_iterator i = cfg.lib_paths().begin(),
                                                     i_end = cfg.lib_paths().end();
         i != i_end;
         ++i)
    {
        YLOG_GLOBAL(info) << "added libpath '" << *i;
        this->libpath_.push_back(*i);
    }
    for (configuration::module_list::const_iterator i = cfg.modules().begin(),
                                                    i_end = cfg.modules().end();
         i != i_end;
         ++i)
    {
        try
        {
            load_module(reactors, *i);
        }
        catch (std::exception const& e)
        {
            throw std::runtime_error(
                "module \"" + i->name + "\" load exception \"" + e.what() + "\"");
        }
    }
}

API_PUBLIC void repository::start()
{
    for (load_map_t::iterator i = this->load_map_.begin(), i_end = this->load_map_.end();
         i != i_end;
         ++i)
    {
        try
        {
            i->wrapper->start();
            i->state = module_state::started;
        }
        catch (std::exception const& e)
        {
            throw std::runtime_error(
                "module \"" + i->name + "\" start exception \"" + e.what() + "\"");
        }
    }
}

API_PUBLIC void repository::reload(const configuration& cfg)
{
    for (configuration::module_list::const_iterator i = cfg.modules().begin(),
                                                    i_end = cfg.modules().end();
         i != i_end;
         ++i)
    {
        find_i(i->name)->reload(i->options.get_child("options"));
    }
}

API_PUBLIC void repository::stop()
{
    for (load_map_t::reverse_iterator i = this->load_map_.rbegin(), i_end = this->load_map_.rend();
         i != i_end;
         ++i)
    {
        if (i->state != module_state::started) continue;

        try
        {
            i->wrapper->stop();
            i->state = module_state::stopped;
        }
        catch (std::exception const& e)
        {
            throw std::runtime_error(
                "module \"" + i->name + "\" stop exception \"" + e.what() + "\"");
        }
    }
}

API_PUBLIC void repository::fini()
{
    for (load_map_t::reverse_iterator i = this->load_map_.rbegin(), i_end = this->load_map_.rend();
         i != i_end;
         ++i)
    {
        if (i->state == module_state::finished) continue;
        i->state = module_state::finished;
        try
        {
            i->wrapper->fini();
            drop_i(i->wrapper->module->name());
        }
        catch (std::exception const& e)
        {
            throw std::runtime_error(
                "module \"" + i->name + "\" fini exception \"" + e.what() + "\"");
        }
    }
    load_map_.clear();
}

void repository::insert_i(const string& __name, const repository::module_wrapper& __optr)
{
    this->load_map_.push_back({ __name, __optr, module_state::inited });
    this->services_.emplace(
        std::piecewise_construct, std::forward_as_tuple(__name), std::forward_as_tuple(__optr));
}

API_PUBLIC repository::module_ptr repository::find(const string& __name)
{
    auto wrapper = this->find_i(__name);
    return wrapper ? wrapper->module : repository::module_ptr();
}

repository::module_wrapper repository::find_i(const string& __name)
{
    map_t::iterator found = this->services_.find(__name);
    if (found == this->services_.end()) return repository::module_wrapper();
    std::lock_guard<spinlock> guard(found->second.lock);
    return found->second.wrapper;
}

void repository::drop_i(const string& __name)
{
    map_t::iterator found = this->services_.find(__name);
    if (found == this->services_.end()) return;
    std::lock_guard<spinlock> guard(found->second.lock);
    found->second.wrapper.reset();
}

API_PUBLIC
std::shared_ptr<repository> repository::instance_ptr()
{
    static std::shared_ptr<repository> repo = std::make_shared<repository>();
    return repo;
}

API_PUBLIC
repository& repository::instance()
{
    return *instance_ptr();
}

}
