#include <yplatform/hash/sha1.h>

namespace yplatform {

namespace detail {

typedef unsigned char id_t[SHA1_HASH_RAWSZ];

static signed char from_hex[] = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 00 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 10 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 20 */
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  -1, -1, -1, -1, -1, -1, /* 30 */
    -1, 10, 11, 12, 13, 14, 15, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 40 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 50 */
    -1, 10, 11, 12, 13, 14, 15, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 60 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 70 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 80 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 90 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* a0 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* b0 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* c0 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* d0 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* e0 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* f0 */
};

static char to_hex[] = "0123456789abcdef";

inline void sha1(id_t& out, const void* data, size_t len)
{
    SHA_CTX c;

    SHA1_Init(&c);
    SHA1_Update(&c, data, len);
    SHA1_Final(out, &c);
}

inline void format_symbol(std::string& str, unsigned int val)
{
    str += to_hex[val >> 4];
    str += to_hex[val & 0xf];
}

inline void format_id(std::string& str, const id_t& id)
{
    for (size_t i = 0; i < sizeof(id); i++)
    {
        format_symbol(str, id[i]);
    }
}

inline int id_from_string(id_t* out, const std::string& str)
{
    size_t p;
    size_t length_estimate = str.length();

    if (length_estimate > SHA1_HASH_HEXSZ)
    {
        length_estimate = SHA1_HASH_HEXSZ;
    }

    if (length_estimate % 2)
    {
        length_estimate--;
    }

    for (p = 0; p < length_estimate; p += 2)
    {
        int v = (from_hex[(unsigned char)str[p + 0]] << 4) | from_hex[(unsigned char)str[p + 1]];

        if (v < 0)
        {
            return 1;
        }

        (*out)[p / 2] = (unsigned char)v;
    }

    for (; p < SHA1_HASH_HEXSZ; p += 2)
    {
        (*out)[p / 2] = 0x0;
    }

    return 0;
}

}

std::string sha1(const std::string& sdata)
{
    detail::id_t id;
    detail::sha1(id, sdata.data(), sdata.size());
    std::string result;
    result.reserve(SHA1_HASH_HEXSZ);
    detail::format_id(result, id);
    return result;
}

}
