#include <yplatform/log/detail/spd.h>
#include <yplatform/log/detail/syslog_resolve_facility.h>
#include <yplatform/log/reopenable.h>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/filesystem.hpp>

namespace yplatform { namespace log { namespace detail { namespace spd {

spdlog::level::level_enum level_from_str(const std::string& name)
{
    using namespace spdlog::level;
    const auto level_name = std::find(std::cbegin(level_names), std::cend(level_names), name);
    if (level_name == std::cend(level_names))
    {
        throw std::logic_error("Unsupported level: " + name);
    }
    return static_cast<level_enum>(level_name - std::cbegin(level_names));
}

template <class Mutex>
class reopenable_file_sink
    : public spdlog::sinks::base_sink<Mutex>
    , public log::reopenable
{
public:
    reopenable_file_sink(const std::string& filename, bool force_flush) : file_helper_(force_flush)
    {
        file_helper_.open(filename);
    }

    void flush() final
    {
        const std::lock_guard<std::mutex> lock(mutex_);
        file_helper_.flush();
    }

    void reopen() final
    {
        const std::lock_guard<std::mutex> lock(mutex_);
        file_helper_.reopen(false);
    }

private:
    std::mutex mutex_;
    spdlog::details::file_helper file_helper_;

    void _sink_it(const spdlog::details::log_msg& msg) final
    {
        const std::lock_guard<std::mutex> lock(mutex_);
        file_helper_.write(msg);
    }
};

using reopenable_file_sink_st = reopenable_file_sink<spdlog::details::null_mutex>;
using reopenable_file_sink_mt = reopenable_file_sink<std::mutex>;
using sinks_map = std::unordered_map<std::string, spdlog::sink_ptr>;

struct named_sink
{
    boost::optional<std::string> name;
    spdlog::sink_ptr value;
};

named_sink make_sink(const sinks_map& other, const log::sink_settings& config, bool async)
{
    int facility = 0;
    boost::filesystem::path parent_path;
    switch (config.type)
    {
    case sink_type::file:
        parent_path = boost::filesystem::path(config.path).parent_path();
        if (!parent_path.empty())
        {
            boost::filesystem::create_directories(parent_path);
        }
        if (async)
        {
            return named_sink{ config.name,
                               std::make_shared<reopenable_file_sink_mt>(
                                   config.path, config.force_flush) };
        }
        else
        {
            return named_sink{ config.name,
                               std::make_shared<reopenable_file_sink_st>(
                                   config.path, config.force_flush) };
        }
    case sink_type::stdout:
        if (async)
        {
            return named_sink{ config.name, std::make_shared<spdlog::sinks::stdout_sink_mt>() };
        }
        else
        {
            return named_sink{ config.name, std::make_shared<spdlog::sinks::stdout_sink_st>() };
        }
    case sink_type::syslog:
        facility = log::detail::syslog_resolve_facility(config.facility);

        return named_sink{
            config.name, std::make_shared<spdlog::sinks::syslog_sink>(config.ident, 0, facility)
        };
    case sink_type::reference:
        return named_sink{ config.name, other.at(config.name.get()) };
    default:
        throw std::logic_error(
            "Unsupported sink type: " + std::to_string(static_cast<int>(config.type)));
    }
}

spdlog::async_overflow_policy make_overflow_policy(const std::string& value)
{
    static std::unordered_map<std::string, spdlog::async_overflow_policy> map = { {
        { "block_retry", spdlog::async_overflow_policy::block_retry },
        { "discard_log_msg", spdlog::async_overflow_policy::discard_log_msg },
    } };
    const auto it = map.find(value);
    if (it == map.end())
    {
        throw std::logic_error("Unsupported overflow policy: " + value);
    }
    return it->second;
}

box make_box(const std::string& name, const log::log_settings& config)
{
    sinks_map named_sinks;
    std::vector<spdlog::sink_ptr> sinks;
    for (const auto& sink_config : config.sinks)
    {
        auto named_sink = make_sink(named_sinks, sink_config, config.async);
        if (named_sink.name)
        {
            named_sinks.insert({ named_sink.name.get(), std::move(named_sink.value) });
        }
        else
        {
            sinks.push_back(std::move(named_sink.value));
        }
    }
    for (auto& v : named_sinks)
    {
        sinks.push_back(std::move(v.second));
    }
    const auto logger =
        config.async ? make_async_logger(name, sinks, config) : make_sync_logger(name, sinks);
    if (config.format != "")
    {
        logger->set_formatter(std::make_shared<spdlog::pattern_formatter>(config.format));
    }
    logger->set_level(level_from_str(config.level));
    return box{ std::move(logger), std::move(sinks) };
}

boxes make(const log::settings& config)
{
    boxes result;
    for (const auto& [name, log_config] : config.logs)
    {
        result.push_back(make_box(name, log_config));
    }
    return result;
}

} // namespace spd
} // namespace detail
} // namespace log
} // namespace yplatform
