#include <yplatform/reactor/io_pool.h>

#include <sys/syscall.h> // gettid
#include <sys/types.h>   // pid_t
#include <sys/prctl.h>   // prctl

namespace yplatform {

io_pool::io_pool(boost::asio::io_service& io, size_t concurrency_hint)
    : io_wrapper_(nullptr), io_(&io), concurrency_(concurrency_hint), external_(true)
{
}

io_pool::io_pool(size_t concurrency_hint)
    : io_wrapper_(new io_service_with_shutdown(concurrency_hint))
    , io_(io_wrapper_)
    , name_("reactor")
    , concurrency_(concurrency_hint)
    , external_(false)
{
}

io_pool::~io_pool()
{
    if (!external_) delete io_wrapper_;
}

boost::asio::io_service* io_pool::io()
{
    return io_;
}

const boost::asio::io_service* io_pool::io() const
{
    return io_;
}

std::vector<boost::thread::id> io_pool::run()
{
    std::vector<boost::thread::id> result;
    if (!external_)
    {
        for (std::size_t i = 0; i < concurrency_; ++i)
        {
            result.push_back(
                workers_.create_thread(boost::bind(&io_pool::run_worker, this, i + 1))->get_id());
        }
    }
    return result;
}

void io_pool::stop()
{
    if (!external_)
    {
        io_->stop();
        workers_.join_all();
        io_wrapper_->shutdown();
        YLOG_LOCAL(debug) << name_ << " stopped";
    }
}

void io_pool::set_name(const std::string& name)
{
    name_ = name;
}

std::size_t io_pool::size() const
{
    return concurrency_;
}

void io_pool::run_worker(std::size_t n)
{
    assert(!external_);
    pid_t tid = 0;
#if defined(SYS_gettid)
    tid = static_cast<pid_t>(::syscall(SYS_gettid));
#endif
    YLOG_LOCAL(debug) << name_ << " worker thread #" << n << " started (tid=" << tid << ')';

    const string thread_name = "io_pool_" + name_;
    prctl(PR_SET_NAME, thread_name.c_str(), NULL, NULL, NULL);

    while (true)
    {
#ifdef YPLATFORM_DEBUG
        io_->run();
        break;
#else
        try
        {
            io_->run();
            break;
        }
        catch (const boost::system::error_code& e)
        {
            YLOG_LOCAL(error) << name_ << " error_code exception in run_thread: " << e.message();
        }
        catch (const std::exception& e)
        {
            YLOG_LOCAL(error) << name_ << " std exception in run_thread: " << e.what();
        }
        catch (...)
        {
            YLOG_LOCAL(error) << name_ << " unknown exception in run_threads";
        }
#endif
    }
    YLOG_LOCAL(debug) << name_ << " worker thread #" << n << " ended (tid=" << tid << ')';
}

}
