#include <yplatform/reactor/reactor.h>

namespace yplatform {

reactor::reactor() : next_io_data_(0), stopped_(false)
{
}

reactor::reactor(std::shared_ptr<io_pool> external_pool) : next_io_data_(0), stopped_(false)
{
    pools_.push_back(shared_ptr_cast<boost::shared_ptr>::from(external_pool));
}

reactor::reactor(boost::asio::io_service& io, size_t concurrency_hint)
    : reactor(std::make_shared<io_pool>(io, concurrency_hint))
{
}

void reactor::init(unsigned pool_count, unsigned thread_count)
{
    cfg_.pool_count = pool_count;
    cfg_.io_threads = thread_count;
    init();
}

void reactor::init(const reactor_cfg& cfg)
{
    cfg_ = cfg;
    init();
}

void reactor::run()
{
    for (std::size_t i = 0; i < cfg_.pool_count; ++i)
    {
        work_.push_back(boost::make_shared<work>(*pools_[i]->io()));
        const auto tids = pools_[i]->run();
        for (auto&& tid : tids)
        {
            threads_pools_map_.insert({ tid, i });
        }
    }
    YLOG_LOCAL(info) << "reactor \"" << name_ << "\" started with " << pools_.size() << " pools x"
                     << cfg_.io_threads << " threads";
}

void reactor::stop()
{
    if (work_.size())
    {
        stopped_ = true;
        work_.clear();
    }
}

void reactor::fini()
{
    for (std::size_t i = 0; i < pools_.size(); ++i)
        pools_[i]->stop();
    threads_pools_map_.clear();
}

const std::string& reactor::name() const
{
    return name_;
}

void reactor::set_name(const std::string& name)
{
    for (std::size_t i = 0; i < pools_.size(); ++i)
        pools_[i]->set_name(name);
    name_ = name;
}

void reactor::set_logger(const yplatform::log::source& lg)
{
    for (std::size_t i = 0; i < pools_.size(); ++i)
        pools_[i]->logger(lg);
}

std::size_t reactor::get_index() const
{
    if (stopped_) throw std::runtime_error("io pool stopped");
    const auto by_thread = threads_pools_map_.find(boost::this_thread::get_id());
    if (by_thread != threads_pools_map_.end())
    {
        return by_thread->second;
    }
    return next_io_data_++ % pools_.size();
}

const io_pool_ptr& reactor::get_pool() const
{
    return pools_[get_index()];
}

const std::vector<io_pool_ptr>& reactor::get_pools() const
{
    return pools_;
}

boost::asio::io_service* reactor::io()
{
    return get_pool()->io();
}

const boost::asio::io_service* reactor::io() const
{
    return get_pool()->io();
}

const io_pool_ptr& reactor::operator[](std::size_t i) const
{
    return pools_[i];
}

size_t reactor::size() const
{
    return pools_.size();
}

bool reactor::plain() const
{
    for (auto& pool : pools_)
    {
        if (pool->size() != 1) return false;
    }
    return true;
}

boost::shared_ptr<reactor> reactor::make_not_owning_copy(reactor& external_reactor)
{
    auto ret = boost::make_shared<reactor>();
    for (auto& pool : external_reactor.pools_)
    {
        ret->pools_.emplace_back(boost::make_shared<io_pool>(*pool->io(), pool->size()));
    }
    ret->threads_pools_map_ = external_reactor.threads_pools_map_;
    ret->set_name(external_reactor.name_);
    if (external_reactor.pools_.size())
    {
        ret->set_logger(external_reactor.pools_.front()->logger());
    }
    return ret;
}

void reactor::init()
{
    for (std::size_t i = 0; i < cfg_.pool_count; ++i)
    {
        pools_.push_back(boost::make_shared<io_pool>(cfg_.io_threads));
    }
}

reactor_ptr reactor_set::add(const std::string& name, const reactor_cfg& cfg)
{
    reactor_ptr new_reactor(new reactor());
    new_reactor->init(cfg);
    new_reactor->set_name(name);
    return add(name, new_reactor);
}

reactor_ptr reactor_set::add(const std::string& name, const reactor_ptr& reactor)
{
    reactors_[name] = reactor;
    if (name == "global") global_ = reactor;
    return reactor;
}

reactor_ptr reactor_set::get(const std::string& name)
{
    if (name == "global")
    {
        if (!global_)
        {
            throw std::runtime_error("reactor \"global\" not found");
        }
        else
        {
            return global_;
        }
    }
    auto it = reactors_.find(name);
    if (it == reactors_.end()) throw std::runtime_error("reactor \"" + name + "\" not found");
    return it->second;
}

bool reactor_set::exists(const std::string& name) const
{
    if (name == "global")
    {
        return !!global_;
    }
    return reactors_.find(name) != reactors_.end();
}

reactor_ptr reactor_set::get_global()
{
    return global_;
}

void reactor_set::run()
{
    for (auto const& pair : reactors_)
        pair.second->run();
}

void reactor_set::stop()
{
    for (auto const& pair : reactors_)
        pair.second->stop();
}

void reactor_set::fini()
{
    for (auto const& pair : reactors_)
        pair.second->fini();
    reactors_.clear();
}

const std::map<std::string, reactor_ptr>& reactor_set::get_all()
{
    return reactors_;
}

}
