#include <yplatform/util/unique_id.h>
#include <boost/thread.hpp>
#include <stdlib.h>
#include <netdb.h>
#include <iomanip>
#include <sstream>
#include <atomic>
#include <sys/syscall.h> // gettid
#include <sys/types.h>

namespace yplatform { namespace util {

namespace {

unsigned char get_local_ip_addr()
{
    hostent* h = gethostent();
    if (!h) return 0;

    return static_cast<unsigned char>(h->h_addr[h->h_length - 1]);
}

const unsigned char LOCAL_IP_ADDRESS = get_local_ip_addr();

const char code_table[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwx";
const size_t code_table_size(sizeof(code_table) / sizeof(*code_table) - 1);

const size_t single_divisor(code_table_size);
const size_t double_divisor(single_divisor* code_table_size);
const size_t triple_divisor(double_divisor* code_table_size);

class number_generator
{
public:
    typedef unsigned int number_type;

    static number_generator& instance()
    {
        static number_generator rnd;
        return rnd;
    }

    number_type operator()()
    {
        return ++counter_;
    }

private:
    number_generator() : counter_(0)
    {
    }

private:
    std::atomic<number_type> counter_;
};

struct simple_date_t
{
    std::uint8_t second;
    std::uint8_t minute;
    std::uint8_t hour;
    std::uint8_t day;
    std::uint8_t month;
    std::uint32_t year;

    simple_date_t(std::time_t time)
    {
        second = std::uint8_t(time % 60);
        time /= 60;
        minute = std::uint8_t(time % 60);
        time /= 60;
        hour = std::uint8_t(time % 24);
        time /= 24;
        day = std::uint8_t(time % 30);
        time /= 30;
        month = std::uint8_t(time % 12);
        time /= 12;
        year = std::uint32_t(time);
    }
};

} // namespace anon

std::string make_unique_id(unsigned long long seed)
{
    std::string result;

    pid_t tid = 0;
#if defined(SYS_gettid)
    tid = static_cast<pid_t>(::syscall(SYS_gettid));
#endif
    const simple_date_t simple_date(std::time(nullptr));

    number_generator::number_type generated_value = number_generator::instance()();

    result.reserve(12);
    result += code_table[simple_date.second];
    result += code_table[simple_date.minute];
    result += code_table
        [static_cast<unsigned long long>(simple_date.hour + simple_date.day + simple_date.month) %
         code_table_size];
    result += code_table[(generated_value) % code_table_size];
    result += code_table[(generated_value / single_divisor) % code_table_size];
    result += code_table[(generated_value / double_divisor) % code_table_size];
    result += code_table[(generated_value / triple_divisor) % code_table_size];
    result += code_table[static_cast<unsigned long long>(getpid()) % code_table_size];
    result += code_table[static_cast<unsigned long long>(tid) % code_table_size];
    result += code_table[seed % code_table_size];
    result += code_table[(seed + code_table_size / 2) % code_table_size];
    result += code_table[LOCAL_IP_ADDRESS % code_table_size];

    return result;
}

} // namespace util
} // namespace yplatform
