#pragma once

#include <stack>

#include <yamail/data/reflection/reflection.h>
#include <boost/property_tree/ptree.hpp>
#include <boost/core/demangle.hpp>

namespace yamail { namespace data { namespace deserialization {

using namespace yamail::data::reflection;
using boost::property_tree::ptree;

namespace property_tree {

struct RootNodeTag {};

namespace detail {

template <typename Derived>
class Reader : public Visitor {
public:
    template <typename T>
    void apply(T& res) {
        applyVisitor( res, *derivedThis(), RootNodeTag() );
    }

    template <typename Value, typename ... Args>
    void onValue(Value & p, NamedItemTag<Args...> tag) {
        p = level().template get<Value>(ptree::path_type(stringName(tag), '\0') );
    }

    template <typename Value>
    void onValue(Value & p, SequenceItemTag) {
        if( iter() == level().end() ) {
            throw std::runtime_error("Nameless items iterator out of range in PtreeReader");
        }
        p = iter()->second.template get_value<Value>();
        ++iter();
    }

    template <typename Struct, typename ... Args>
    Derived onStructStart(Struct& , NamedItemTag<Args...> tag) {
        return Derived( level().get_child(ptree::path_type(name(tag), '\0')) );
    }

    template <typename Struct>
    Derived onStructStart(Struct& , RootNodeTag) { return *derivedThis(); }

    template <typename Struct>
    Derived onStructStart(Struct& , SequenceItemTag) {
        return Derived( (iter()++)->second );
    }

    template <typename Map, typename Tag>
    auto onMapStart(Map & p, Tag tag)
            -> std::enable_if_t<!std::is_same<std::string, typename Map::value_type>::value, Derived> {
        auto retval = onStructStart(p, tag);
        for( const auto & i : retval.level()) {
            typename Map::key_type key;
            if (!boost::conversion::try_lexical_convert(i.first, key)) {
                throw std::runtime_error("Failed to convert \"" + i.first + "\" to "
                                         + boost::core::demangle(typeid(key).name()));
            }
            p[key];
        }
        return retval;
    }

    template <typename Map, typename Tag>
    auto onMapStart(Map & p, Tag tag)
            -> std::enable_if_t<std::is_same<std::string, typename Map::value_type>::value, Derived> {
        auto retval = onStructStart(p, tag);
        for( const auto & i : retval.level()) {
            p[i.first];
        }
        return retval;
    }

    template <typename Sequence, typename Tag>
    Derived onSequenceStart(Sequence & p, Tag tag) {
        auto retval = onStructStart(p, tag);
        p.resize( retval.level().size() );
        return std::move(retval);
    }

    template <typename Sequence, std::size_t N, typename Tag>
    Derived onSequenceStart(Sequence (& p)[N], Tag tag) {
        return onStructStart(p, tag);
    }

    template <template <class> class Optional, typename P, typename Tag>
    auto onOptional(Optional<P> & p, Tag tag)
            -> std::enable_if_t<!std::is_arithmetic<P>::value && is_optional<Optional<P>>::value, bool> {
        return onOptionalImpl(p, tag);
    }

    template <template <class> class Optional, typename P, typename Tag>
    auto onOptional(Optional<P> & p, Tag tag)
            -> std::enable_if_t<std::is_arithmetic<P>::value && is_optional<Optional<P>>::value, bool> {
        return onOptionalIntegral(p, tag);
    }

    template<typename Pointer, typename ... Args>
    bool onSmartPointer(Pointer& p, NamedItemTag<Args...> tag) {
        const bool fieldFound = ( level().find( name(tag) ) != level().not_found() );
        if( fieldFound ) {
            p.reset(Access::constructPtr<typename Pointer::element_type>());
        }
        return fieldFound;
    }

    template<typename Pointer>
    bool onSmartPointer(Pointer& p, RootNodeTag) {
        p.reset(Access::constructPtr<typename Pointer::element_type>());
        return true;
    }

    template<typename Pointer>
    bool onSmartPointer(Pointer& p, SequenceItemTag) {
        const bool fieldFound = !level().empty();
        if( fieldFound ) {
            p.reset(Access::constructPtr<typename Pointer::element_type>());
        }
        return fieldFound;
    }

    template<typename Tag>
    void onPtree(ptree& p, Tag tag) {
        p = onStructStart(p, tag).level();
        onStructEnd(p, tag);
    }

private:
    const ptree & level() const { return derivedThis()->level(); }
    ptree::const_iterator & iter() { return derivedThis()->iter(); }
    const ptree::const_iterator & iter() const { return derivedThis()->iter(); }

    Derived* derivedThis() {
        return static_cast<Derived*>(this);
    }

    const Derived* derivedThis() const {
        return static_cast<const Derived*>(this);
    }

    template <template <class> class Optional, typename P, typename ... Args>
    bool onOptionalIntegral(Optional<P> & p, NamedItemTag<Args...> tag) {
        const bool optFieldFound = ( level().find( name(tag) ) != level().not_found() );
        const bool hasValue = optFieldFound && !level().get_child(ptree::path_type(name(tag), '\0')).data().empty();
        if( hasValue ) {
            p = Access::construct<P>();
        }
        return hasValue;
    }

    template <template <class> class Optional, typename P>
    bool onOptionalIntegral(Optional<P> & p, SequenceItemTag) {
        const bool optFieldFound = !level().empty();
        const bool hasValue = optFieldFound;
        if( hasValue ) {
            p = Access::construct<P>();
        }
        return hasValue;
    }

    template <template <class> class Optional, typename P, typename ... Args>
    bool onOptionalImpl(Optional<P> & p, NamedItemTag<Args...> tag) {
        const bool optFieldFound = (level().find( name(tag) ) != level().not_found());
        if( optFieldFound ) {
            p = Access::construct<P>();
        }
        return optFieldFound;
    }

    template <template <class> class Optional, typename P>
    bool onOptionalImpl(Optional<P> & p, RootNodeTag) {
        p = Access::construct<P>();
        return true;
    }

    template <template <class> class Optional, typename P>
    bool onOptionalImpl(Optional<P> & p, SequenceItemTag) {
        const bool optFieldFound = !level().empty();
        if( optFieldFound ) {
            p = Access::construct<P>();
        }
        return optFieldFound;
    }
};

}

class Reader : public detail::Reader<Reader> {
public:
    explicit Reader(const ptree& pt) : level_(pt), iter_(level().begin()) {
    }

private:
    friend class property_tree::detail::Reader<Reader>;

    const ptree & level() const { return level_; }
    ptree::const_iterator & iter() { return iter_; }
    const ptree::const_iterator & iter() const { return iter_; }

    const ptree& level_;
    ptree::const_iterator iter_;
};

} // namespace property_tree

template <typename T>
inline void fromPtree(const ptree& p, T& v) {
    property_tree::Reader(p).apply(v);
}

template <typename T>
inline T fromPtree(const ptree& p) {
    T retval = Access::construct<T>();
    fromPtree(p, retval);
    return std::move(retval);
}

}}}
