#pragma once

#include <yamail/data/reflection.h>
#include <yamail/data/reflection/visitor.h>
#include <yamail/data/reflection/details/urlencoded_traits.h>

#include <boost/range/algorithm/transform.hpp>
#include <boost/lexical_cast.hpp>
#include <spdlog/details/format.h>


namespace yamail::data::deserialization {

namespace urlencoded {
using namespace yamail::data::reflection;

struct ReaderException: public std::runtime_error {
    using std::runtime_error::runtime_error;
};

struct NoSuchEntry: public ReaderException {
    NoSuchEntry(std::string_view name)
        : ReaderException(fmt::format("No such entry: {}", name))
    { }
};

struct BadCast: public ReaderException {
    BadCast(std::string_view name, std::string_view value)
        : ReaderException(fmt::format("Bad cast of {} with value {}", name, value))
    { }
};

inline std::optional<std::vector<std::string>> getManyWithException(const std::string&) {
    throw ReaderException("Cannot get many elements");
}

using StringToBool = std::function<bool(const std::string&, bool&)>;

inline bool yesNoToString(const std::string& str, bool& val) {
    if (str == "yes" || str == "1" || str == "true") {
        val = true;
        return true;
    } else if (str == "no" || str == "0" || str == "false") {
        val = false;
        return true;
    } else {
        return false;
    }
}

template<class Options>
struct Reader: public Visitor {
    const Options& options;

    Reader(const Options& o)
        : options(o)
    { }

    template<typename Map, typename Tag>
    Visitor onMapStart(Map&& , Tag) = delete;

    template<typename Value, typename ... Args>
    void onValue(Value& v, NamedItemTag<Args...> tag) {
        using RealValue = unwrap_t<Value>;

        const auto str = options.template one<Value>(name(tag));
        if (!str) {
            throw NoSuchEntry(name(tag));
        }

        try {
            if constexpr (std::is_same_v<RealValue, bool>) {
                if (!options.stringToBool(*str, v)) {
                    throw BadCast(name(tag), *str);
                }
            } else if constexpr (std::is_unsigned_v<RealValue> || std::is_same_v<RealValue, std::time_t>) {
                auto temp = std::stoll(*str);
                if (temp < 0) {
                    throw BadCast(name(tag), *str);
                }
                v = static_cast<RealValue>(temp);
            } else if constexpr (std::is_pointer_v<RealValue>) {
                if (!v) {
                    if (options.initEmptyRawPointer) {
                        v = Access::constructPtr<std::remove_pointer_t<RealValue>>();
                    } else {
                        throw ReaderException(std::string("cannot set to empty pointer: ") + name(tag));
                    }
                }
                *v = boost::lexical_cast<std::remove_pointer_t<RealValue>>(*str);
            } else {
                v = boost::lexical_cast<RealValue>(*str);
            }
        } catch (const std::logic_error&) {
            throw BadCast(name(tag), *str);
        } catch(const boost::bad_lexical_cast&) {
            throw BadCast(name(tag), *str);
        }
    }

    template <typename Sequence, typename Named>
    Visitor onSequenceStart(Sequence& c, Named tag) {
        static_assert(!unwrap_v<Sequence>, "Sequence type must not be wrapped");

        using Value = std::decay_t<decltype(*std::begin(c))>;

        const auto values = options.template many<Value>(name(tag));

        if (!values) {
            throw NoSuchEntry(name(tag));
        }

        boost::transform(*values, std::back_inserter(c), [&] (const std::string& val) {
            try {
                return boost::lexical_cast<Value>(val);
            }  catch(const boost::bad_lexical_cast&) {
                throw BadCast(name(tag), val);
            }
        });

        return Visitor();
    }

    template<typename Struct, typename ... Args>
    Reader onStructStart(Struct&, NamedItemTag<Args...>) { return *this; }

    template<template<class> class Opt, class Value, class Named>
    bool onOptional(Opt<Value>& p, Named tag) {
        if constexpr (boost::fusion::traits::is_sequence<unwrap_t<Value>>::value) {
            static_assert(!unwrap_v<Value>, "Optional value type must not be wrapped");

            try {
                Value retval = Access::construct<Value>();
                Reader reader = *this;
                applyVisitor(retval, reader, namedItemTag("root"));
                p = std::move(retval);
            } catch (const NoSuchEntry&) { }

            return false;
        } else {
            const auto val = options.template one<Value>(name(tag));
            if (val) {
                p = Access::construct<unwrap_t<Value>>();
            }
            return static_cast<bool>(val);
        }
    }

    template<typename Pointer, typename Tag>
    bool onSmartPointer(Pointer& p, Tag tag) {
        using Value = typename Pointer::element_type;

        if constexpr (boost::fusion::traits::is_sequence<unwrap_t<Value>>::value) {
            static_assert(!unwrap_v<Value>, "Smart pointer element type must not be wrapped");

            try {
                auto retval = std::make_unique<Value>(Access::construct<Value>());
                Reader reader = *this;
                applyVisitor(*retval, reader, namedItemTag("root"));
                p.reset(retval.release());
            } catch (const NoSuchEntry&) { }

            return false;
        } else {
            const auto val = options.template one<Value>(name(tag));
            if (val) {
                p.reset(Access::constructPtr<Value>());
            }
            return static_cast<bool>(val);
        }
    }
};

template<class GetOpt, class GetOpts>
struct DefaultOptions {
    const GetOpt& getOne;
    const GetOpts& getMany;
    const StringToBool& stringToBool;
    bool initEmptyRawPointer;

    template<class Value>
    auto one(const std::string& str) const {
        return getOne(str);
    }

    template<class Value>
    auto many(const std::string& str) const {
        return getMany(str);
    }
};

}

template <typename T, class S>
void fromUrlencodedWithOptions(T& retval, const S& opts) {
    urlencoded::Reader reader(opts);
    reflection::applyVisitor(retval, reader, reflection::namedItemTag("root"));
}

template <typename T, class O, class M>
void fromUrlencoded(T& retval, const O& one, const M& many) {
    fromUrlencodedWithOptions(retval, urlencoded::DefaultOptions<O, M> {
        .getOne=one,
        .getMany=many,
        .stringToBool=urlencoded::yesNoToString,
        .initEmptyRawPointer=false
    });
}

template <typename T, class O>
void fromUrlencoded(T& retval, const O& one) {
    fromUrlencodedWithOptions(retval, urlencoded::DefaultOptions<O, decltype(urlencoded::getManyWithException)> {
        .getOne=one,
        .getMany=urlencoded::getManyWithException,
        .stringToBool=urlencoded::yesNoToString,
        .initEmptyRawPointer=false
    });
}

}
