#pragma once

#include <stack>

#include <boost/property_tree/ptree.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/shared_ptr.hpp>

#ifdef ARCADIA_BUILD
#include <contrib/libs/yajl/api/yajl_parse.h>
#else
#include <yajl/yajl_parse.h> // Y_IGNORE
#endif

#include <yamail/data/reflection/reflection.h>
#include <yamail/data/deserialization/ptree_reader.h>
#include <yamail/data/reflection/options.h>

namespace yamail { namespace data { namespace deserialization {

using namespace yamail::data::reflection;
namespace json {

struct JsonToPtreeException: public std::runtime_error {
    JsonToPtreeException(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

namespace json2ptree {

using boost::property_tree::ptree;

struct OptForceNull : std::true_type {};

constexpr auto optForceNull = OptForceNull{};

template<typename Options>
class Parser {
public:
    Parser(Options options)
            : options_(std::move(options))
            , result_(new ptree) {
    }

    void parse(const std::string& str) {
        yajl_callbacks callbacks = {
            onNullCallback,
            onBooleanCallback,
            onIntegerCallback,
#ifdef ARCADIA_BUILD
            onUnsignedIntegerCallback,
#endif
            onDoubleCallback,
            onNumberCallback,
            onStringCallback,
            onStartMapCallback,
            onMapKeyCallback,
            onEndContainerCallback,
            onStartArrayCallback,
            onEndContainerCallback
        };

        yajl_handle handlers = yajl_alloc(&callbacks, 0, this);
        const unsigned char* strCasted = reinterpret_cast<const unsigned char*>(str.c_str());
        yajl_status res = yajl_parse(handlers, strCasted, str.length());
        if (res != yajl_status_ok) {
            unsigned char* err = yajl_get_error(handlers, 1, strCasted, str.length());
            const std::string errStr(reinterpret_cast<char*>(err));
            yajl_free_error(handlers, err);
            yajl_free(handlers);
            throw JsonToPtreeException("yajl_parse failed: " + errStr);
        }
        res = yajl_complete_parse(handlers);
        if (res != yajl_status_ok) {
            unsigned char* err = yajl_get_error(handlers, 1, strCasted, str.length());
            const std::string errStr(reinterpret_cast<char*>(err));
            yajl_free_error(handlers, err);
            yajl_free(handlers);
            throw JsonToPtreeException("yajl_complete_parse failed: " + errStr);
        }
        yajl_free(handlers);
    }

    const ptree& tree() const {
        return *result_;
    }

private:
    int onNull() {
        ptree child;
        if( frames_.empty() ) {
            return 0;
        }

        if constexpr (!hasOption<OptForceNull, Options>()) {
            frames_.top().addEntity( child );
        }

        return 1;
    }

    int onString(const std::string& val) {
        ptree child(val);
        if( frames_.empty() ) {
            return 0;
        }
        frames_.top().addEntity( child );
        return 1;
    }

    int onStartMap() {
        Frame frame = {PtreePtr(new ptree), "", true};
        frames_.push(frame);
        return 1;
    }

    int onMapKey(const std::string& key) {
        if( frames_.empty() ) {
            return 0;
        }
        frames_.top().key = key;
        return 1;
    }

    int onStartArray() {
        Frame frame = {PtreePtr(new ptree), "", false};
        frames_.push(frame);
        return 1;
    }

    int onEndContainer() {
        if( frames_.empty() ) {
            return 0;
        }
        const Frame oldFrame = frames_.top();
        frames_.pop();
        if (!frames_.empty()) {
            frames_.top().addEntity(*oldFrame.tree);
        } else {
            result_ = oldFrame.tree;
        }
        return 1;
    }

    static int onNullCallback(void* ctx) {
        return static_cast<Parser*>(ctx)->onNull();
    }

    static int onBooleanCallback(void *ctx, int booleanVal) {
        return static_cast<Parser*>(ctx)->onString(
                boost::lexical_cast<std::string>(booleanVal)
        );
    }

    static int onIntegerCallback(void *ctx, long long integerVal) {
        return static_cast<Parser*>(ctx)->onString(
                boost::lexical_cast<std::string>(integerVal)
                );
    }

    #ifdef ARCADIA_BUILD
    static int onUnsignedIntegerCallback(void *ctx, unsigned long long integerVal) {
        return static_cast<Parser*>(ctx)->onString(
                boost::lexical_cast<std::string>(integerVal)
                );
    }
    #endif

    static int onDoubleCallback(void *ctx, double doubleVal) {
        return static_cast<Parser*>(ctx)->onString(
                boost::lexical_cast<std::string>(doubleVal)
        );
    }

    static int onStringCallback(void * ctx, const unsigned char* stringVal, size_t stringLen) {
        return static_cast<Parser*>(ctx)->onString(
                std::string(reinterpret_cast<const char*>(stringVal), stringLen)
        );
    }

    static int onNumberCallback(void * ctx, const char* numberVal, size_t numberLen) {
        return static_cast<Parser*>(ctx)->onString(
                std::string(numberVal, numberLen)
        );
    }

    static int onStartMapCallback(void* ctx) {
        return static_cast<Parser*>(ctx)->onStartMap();
    }

    static int onMapKeyCallback(void* ctx, const unsigned char* stringVal, size_t stringLen) {
        return static_cast<Parser*>(ctx)->onMapKey(
                std::string(reinterpret_cast<const char*>(stringVal), stringLen)
        );
    }

    static int onStartArrayCallback(void* ctx) {
        return static_cast<Parser*>(ctx)->onStartArray();
    }

    static int onEndContainerCallback(void* ctx) {
        return static_cast<Parser*>(ctx)->onEndContainer();
    }

    typedef boost::shared_ptr<ptree> PtreePtr;
    struct Frame {
        ~Frame() {}

        void addEntity( const ptree& entity) {
            if (isInMapContext) {
                tree->push_back(std::make_pair(key, entity));
            } else {
                tree->push_back(std::make_pair("", entity));
            }
        }
        PtreePtr tree;
        std::string key;
        bool isInMapContext;
    };

    Options options_;
    std::stack<Frame> frames_;
    PtreePtr result_;
};

} // namespace json2ptree

template<typename... Ts>
inline boost::property_tree::ptree toPtree(const std::string& json, std::tuple<Ts...> options =
        std::tuple<>{}) {
    json2ptree::Parser<std::tuple<Ts...>> parser{std::move(options)};
    parser.parse(json);
    return parser.tree();
}

class Reader : public Visitor {};

} // namespace json

template<typename T, typename... Ts>
inline void fromJson(const std::string& json, T& v, std::tuple<Ts...> options = std::tuple<>{}) {
    auto tree = json::toPtree(json, std::move(options));
    fromPtree(tree, v);
}

template<typename T, typename... Ts>
inline T fromJson(const std::string& json, std::tuple<Ts...> options = std::tuple<>{}) {
    T retval = Access::construct<T>();
    fromJson(json, retval, std::move(options));
    return retval;
}

}}}
