#pragma once

#include <yamail/data/reflection/reflection.h>

#include <yaml-cpp/yaml.h>

namespace yamail {
namespace data {
namespace deserialization {

using namespace yamail::data::reflection;

namespace yaml {

struct RootNodeTag {};

class Exception final : public std::exception {
public:
    explicit Exception(std::string message) noexcept
        : message(std::move(message))
    {}

    char const* what() const noexcept override {
        return message.c_str();
    }

private:
    std::string message;
};

class Reader : public Visitor {
public:
    explicit Reader(const YAML::Node& node)
        : node_(node),
          iter(node_.begin())
    {}

    template <typename T>
    void apply(T& res) {
        applyVisitor(res, *this, RootNodeTag());
    }

    template <typename Value, typename... Args>
    void onValue(Value& value, NamedItemTag<Args...> tag) {
        value = to<Value>(findNode(tag));
    }

    template <typename Value>
    void onValue(Value& value, SequenceItemTag) {
        value = to<Value>(currentSeqNode());
        ++iter;
    }

    template <typename Value>
    void onValue(Value& value, RootNodeTag) {
        value = to<Value>(node());
    }

    template <typename Struct, typename... Args>
    Reader onStructStart(Struct&, NamedItemTag<Args...> tag) {
        return Reader(findNode(name(tag)));
    }

    template <typename Struct>
    Reader onStructStart(Struct&, RootNodeTag) {
        return Reader(node());
    }

    template <typename Struct>
    Reader onStructStart(Struct&, SequenceItemTag) {
        const auto& elementNode = currentSeqNode();
        ++iter;
        return Reader(elementNode);
    }

    template <typename Map, typename... Args>
    Reader onMapStart(Map& map, NamedItemTag<Args...> tag) {
        return onMapStart(map, findNode(name(tag)));
    }

    template <typename Map>
    Reader onMapStart(Map& map, SequenceItemTag) {
        return onMapStart(map, currentSeqNode());
    }

    template <typename Sequence, typename... Args>
    Reader onSequenceStart(Sequence& seq, NamedItemTag<Args...> tag) {
        return onSequenceStart(seq, findNode(name(tag)));
    }

    template <typename Sequence>
    Reader onSequenceStart(Sequence& seq, SequenceItemTag) {
        return onSequenceStart(seq, currentSeqNode());
    }

    template <template <class> class Optional, typename T, typename... Args>
    auto onOptional(Optional<T>& value, NamedItemTag<Args...> tag)
            -> std::enable_if_t<is_optional<Optional<T>>::value, bool> {
        const bool valuePresent = nodeExists(name(tag));
        if (valuePresent) {
            value = Access::construct<T>();
        }
        return valuePresent;
    }

    template <template <class> class Optional, typename T>
    auto onOptional(Optional<T>& value, SequenceItemTag)
            -> std::enable_if_t<is_optional<Optional<T>>::value, bool> {
        const bool valuePresent = node().size() > 0;
        if (valuePresent) {
            value = Access::construct<T>();
        }
        return valuePresent;
    }

    template <template <class> class Optional, typename T>
    auto onOptional(Optional<T>& value, RootNodeTag)
            -> std::enable_if_t<is_optional<Optional<T>>::value, bool> {
        value = Access::construct<T>();
        return true;
    }

    template<typename Pointer, typename... Args>
    bool onSmartPointer(Pointer& p, NamedItemTag<Args...> tag) {
        const bool valuePresent = nodeExists(name(tag));
        if (valuePresent) {
            p.reset(Access::constructPtr<typename Pointer::element_type>());
        }
        return valuePresent;
    }

    template<typename Pointer>
    bool onSmartPointer(Pointer& p, SequenceItemTag) {
        const bool valuePresent = node().size() > 0;
        if (valuePresent) {
            p.reset(Access::constructPtr<typename Pointer::element_type>());
        }
        return valuePresent;
    }

    template<typename Pointer>
    bool onSmartPointer(Pointer& p, RootNodeTag) {
        p.reset(Access::constructPtr<typename Pointer::element_type>());
        return true;
    }

private:
    template <typename... Args>
    YAML::Node findNode(NamedItemTag<Args...> tag) const {
        return findNode(name(tag));
    }

    template <typename Name>
    YAML::Node findNode(Name name) const {
        return findNode(std::to_string(name).c_str());
    }

    YAML::Node findNode(const char* name) const {
        const auto valueNode = node()[name];
        if (!valueNode) {
            throw Exception(std::string("Field ") + name + " not found");
        }
        return valueNode;
    }

    YAML::Node currentSeqNode() const {
        if (iter == node().end()) {
            throw Exception("Unexpected end of sequence");
        }
        return *iter;
    }

    bool nodeExists(const char* name) const {
        return node()[name];
    }

    template <typename Map>
    static Reader onMapStart(Map& map, const YAML::Node& mapNode) {
        for (auto i = mapNode.begin(); i != mapNode.end(); ++i) {
            map[to<typename Map::key_type>(i->first)];
        }
        return Reader(mapNode);
    }

    template <typename Sequence>
    static Reader onSequenceStart(Sequence& seq, const YAML::Node& seqNode) {
        seq.resize(seqNode.size());
        return Reader(seqNode);
    }

    template <typename T>
    static T to(const YAML::Node& n) {
        return n.as<T>();
    }

    const YAML::Node& node() const {
        return node_;
    }

    YAML::Node node_;
    YAML::iterator iter;
};

} // yaml namespace

template <typename T>
T& fromYaml(std::istream& stream, T& value) {
    const YAML::Node root = YAML::Load(stream);
    if (!root) {
        throw yaml::Exception("YAML input stream is empty");
    }

    yaml::Reader(root).apply(value);
    return value;
}

template <typename T>
T& fromYaml(const std::string& yamlString, T& value) {
    std::stringstream stream;
    stream << yamlString;
    return fromYaml(stream, value);
}

template <typename T>
std::enable_if_t<std::is_default_constructible<T>::value, T>
fromYaml(const std::string& yamlString) {
    T value = Access::construct<T>();
    fromYaml(yamlString, value);
    return value;
}

template <typename T>
std::enable_if_t<std::is_default_constructible<T>::value, T>
fromYaml(std::istream& stream) {
    T value = Access::construct<T>();
    fromYaml(stream, value);
    return value;
}

}}}
