#pragma once

#include <yamail/data/reflection/details/type_traits.h>

namespace yamail::data::reflection {

/**
 * @brief Default apply_visitor implementation
 *
 * This structure implements default application of default Visitor for type T.
 * There are different specializations for different types and concepts are exist.
 * E.g. for std::pair, maps, containers like vector and list, sequences like array
 * or tuple, and so on.
 *
 * @tparam T --- type to which such algorithm applicable
 * @tparam <anonymous> --- customization point for type-dependend logic
 */
template <typename T, typename = std::void_t<>>
struct apply_visitor_default_impl {
    template <typename Value, typename Visitor, typename Tag>
    static void apply(Visitor& v, Value&& value, Tag&& tag) {
        v.onValue(std::forward<Value>(value), std::forward<Tag>(tag));
    };
};

/**
 * @brief Implementation of visitor to value application strategy
 *
 * This template allows to specialize application of a Visitor to a value with type T.
 * In other words this is the place where interaction between value and Visitor can be
 * customized.
 *
 * @tparam T --- type of value to apply Visitor to
 * @tparam Visitor --- type of Visitor
 */
template <typename T, typename Visitor>
struct apply_visitor_impl : apply_visitor_default_impl<T> {};

template <typename T, typename Visitor, typename Tag>
inline void apply_visitor(Visitor& visitor, T&& value, Tag&& tag ) {
    using impl = apply_visitor_impl<std::remove_reference_t<T>, remove_cvref_t<Visitor>>;
    impl::apply(visitor, std::forward<T>(value), std::forward<Tag>(tag));
}

template <typename T, typename Visitor, typename Tag>
inline void applyVisitor(T& t, Visitor& v, Tag tag ) {
    yamail::data::reflection::apply_visitor(v, t, tag);
}

} // namespace yamail::data::reflection
