#pragma once

#include <yamail/data/reflection/details/type_traits.h>
#include <boost/preprocessor/tuple/remove.hpp>
#include <boost/fusion/adapted.hpp>

namespace yamail::data::reflection {

/**
 * This function can be used to generate run-time error while trying
 * to deserialize read-only property of type reflection
 */
template <typename T1, typename T2>
inline void readOnly(T1&&, T2&&) {
    throw std::logic_error("this property is read-only");
}

/**
 * This function can be used to silently skip deserialization of read-only
 * property of type reflection
 */
template <typename T1, typename T2>
inline void ignore(T1&&, T2&&) {}

/**
 * This class can be used to make a ATD view just like adapted structure
 * view to make a different reflection of the same type T. Tag is used
 * to make able do different views of the same type.
 * It can be used like:
 *
 * using envelope = adt_view<macs::Envelope, my_api_tag>;
 *
 * YREFLECTION_ADAPT_ADT(hound::v2::changes::delta::envelope,
 *    (mid, obj->mid(), read_only(obj, val))
 *    (fid, obj->fid(), read_only(obj, val))
 *    (threadId, obj->threadId(), read_only(obj, val))
 *    (date, obj->date(), read_only(obj, val))
 *
 * ...
 *
 * )
 */
template <typename T, typename Tag>
struct AdtView {
    T obj;
    AdtView(T obj) : obj(std::move(obj)) {}
    const T* operator -> () const { return &obj; }
    T* operator -> () { return &obj; }
    const T& operator * () const { return obj; }
    T& operator * () { return obj; }
};

template <typename T>
struct is_adt_view : std::false_type {};

template <typename T, typename Tag>
struct is_adt_view<AdtView<T, Tag>> : std::true_type {};

template <typename T>
constexpr decltype(auto) adtUnwrap(T&& v,
        std::enable_if_t<!is_adt_view<std::decay_t<T>>::value>* = nullptr) {
    return v;
}

template <typename T>
constexpr decltype(auto) adtUnwrap(T&& v,
        std::enable_if_t<is_adt_view<std::decay_t<T>>::value>* = nullptr) {
    return *v;
}

} // namespace yamail::data::reflection

#define YR_INT_ADT_SET_MEMBER_NAME(R, CLASSNAME, IDX, MEMBERNAME)\
        template <> struct struct_member_name<CLASSNAME, IDX> {\
            typedef char const *type; \
            static type call() { return BOOST_PP_STRINGIZE(MEMBERNAME); } \
        };

#define YR_ADT_MEMBER_NAMES(CLASSNAME, NAMES)\
    namespace boost { namespace fusion { namespace extension {\
    BOOST_PP_SEQ_FOR_EACH_I_R(1,YR_INT_ADT_SET_MEMBER_NAME, CLASSNAME, NAMES) \
    } } }

#define YR_INT_REMOVE_FIRST_ATTR(R, DATA, ATTRS) BOOST_PP_TUPLE_REMOVE(ATTRS, 0)

#define YR_ADAPT_ADT_FILLER_0(...)\
    ((__VA_ARGS__)) YR_ADAPT_ADT_FILLER_1
#define YR_ADAPT_ADT_FILLER_1(...)\
    ((__VA_ARGS__)) YR_ADAPT_ADT_FILLER_0
#define YR_ADAPT_ADT_FILLER_0_END
#define YR_ADAPT_ADT_FILLER_1_END

#define YR_INT_GET_NAME(R, DATA, ATTRS) (BOOST_PP_TUPLE_ELEM(0, ATTRS))

#define YR_INT_FOR_SEQUENCE(SEQ, MACRO)                                 \
    BOOST_PP_IF(                                                        \
        BOOST_PP_DEC(BOOST_PP_SEQ_SIZE(SEQ)),                           \
        BOOST_PP_SEQ_FOR_EACH,                                          \
        BOOST_PP_TUPLE_EAT(5))(                                         \
            MACRO,                                                      \
            0,                                                          \
            BOOST_PP_SEQ_TAIL(SEQ))                                     \

#define YR_INT_FOR_ATTRS(ATTRIBUTES, MACRO)\
        YR_INT_FOR_SEQUENCE(\
                BOOST_PP_CAT(YR_ADAPT_ADT_FILLER_0(0,0,0,0,0)ATTRIBUTES,_END),\
                MACRO)

#define YR_INT_GET_NAMES(ATTRIBUTES)\
        YR_INT_FOR_ATTRS(ATTRIBUTES, YR_INT_GET_NAME)

#define YR_INT_GET_ADT_ATTRS(ATTRIBUTES)\
        YR_INT_FOR_ATTRS(ATTRIBUTES, YR_INT_REMOVE_FIRST_ATTR)


/**
 * BOOST_FUSION_ADAPT_ADT proxy macro which allow to point refelection name of every entry.
 *
 * YREFLECTION_ADAPT_ADT(struct_or_calss_name,
 *      (member1_name, [getter1_return_type, setter1_argument_type], getter1_code, setter1_code)
 *      ...
 *      (memberN_name, [getterN_return_type, setterN_argument_type], getterN_code, setterN_code)
 * );
 */
#define YREFLECTION_ADAPT_ADT(NAME, ATTRIBUTES)\
        BOOST_FUSION_ADAPT_ADT(NAME, YR_INT_GET_ADT_ATTRS(ATTRIBUTES))\
        YR_ADT_MEMBER_NAMES(NAME, YR_INT_GET_NAMES(ATTRIBUTES))

// Setter macro for an attribute named NAME
#define YREFLECTION_MEMBER_SETTER(NAME)\
    yamail::data::reflection::adtUnwrap(obj).NAME=val

// Getter macro for an attribute named NAME
#define YREFLECTION_MEMBER_GETTER(NAME)\
    yamail::data::reflection::forward_ref(\
        yamail::data::reflection::adtUnwrap(obj).NAME\
    )

// Setter macro for NAME(val)-like  setter method
#define YREFLECTION_PROPERTY_SETTER(NAME)\
    yamail::data::reflection::adtUnwrap(obj).NAME(val)

// Getter macro for NAME()-like  getter method
#define YREFLECTION_PROPERTY_GETTER(NAME)\
    yamail::data::reflection::forward_ref(\
        yamail::data::reflection::adtUnwrap(obj).NAME()\
    )

// Setter macro for 'set_name(val)'-like setter method
#define YREFLECTION_SETTER_(name)\
    yamail::data::reflection::adtUnwrap(obj).set_##name(val)

// Getter macro for 'get_name()'-like getter method
#define YREFLECTION_GETTER_(name)\
    yamail::data::reflection::forward_ref(\
        yamail::data::reflection::adtUnwrap(obj).get_##name()\
    )

// Setter macro for 'setNAME(val)'-like setter method
#define YREFLECTION_SETTER(NAME)\
    yamail::data::reflection::adtUnwrap(obj).set##NAME(val)

// Getter macro for 'getNAME()'-like getter method
#define YREFLECTION_GETTER(NAME)\
    yamail::data::reflection::forward_ref(\
        yamail::data::reflection::adtUnwrap(obj).get##NAME()\
    )

// Fictitious setter for read-only reflection of the member
#define YREFLECTION_RO_SETTER(...)\
    yamail::data::reflection::readOnly(obj, val)


#define YREFLECTION_AUTO_MEMBER_IMPL_4(REFLECTION_NAME, NAME, GETTER, SETTER)\
    (REFLECTION_NAME, GETTER(NAME), SETTER(NAME))

#define YREFLECTION_AUTO_MEMBER_IMPL_3(NAME, GETTER, SETTER)\
    YREFLECTION_AUTO_MEMBER_IMPL_4(NAME, NAME, GETTER, SETTER)

/**
 * This macro reflects member NAME with NAME or REFLECT_NAME.
 * Member type will be deduced automatically.
 *
 * YREFLECTION_AUTO_MEMBER([REFLECT_NAME,] NAME, GETTER, SETTER)
 *
 * Parameters:
 *   GETTER       - getter macro which accepts member name as parameter
 *   SETTER       - setter macro which accepts member name as parameter
 *   NAME         - member name and reflection name if REFLECT_NAME is omitted
 *   REFLECT_NAME - name for reflection if specified
 */
#define YREFLECTION_AUTO_MEMBER(...)\
    BOOST_PP_OVERLOAD(YREFLECTION_AUTO_MEMBER_IMPL_,__VA_ARGS__)(__VA_ARGS__)

//=================================================================================
// PRESETS FOR YREFLECTION_AUTO_MEMBER MACRO
//=================================================================================

#define YREFLECTION_AUTO_ATTR(...)\
    YREFLECTION_AUTO_MEMBER(__VA_ARGS__, YREFLECTION_MEMBER_GETTER, YREFLECTION_MEMBER_SETTER)

#define YREFLECTION_AUTO_ATTR_RO(...)\
    YREFLECTION_AUTO_MEMBER(__VA_ARGS__, YREFLECTION_MEMBER_GETTER, YREFLECTION_RO_SETTER)

#define YREFLECTION_AUTO_PROP(...)\
    YREFLECTION_AUTO_MEMBER(__VA_ARGS__, YREFLECTION_PROPERTY_GETTER, YREFLECTION_PROPERTY_SETTER)

#define YREFLECTION_AUTO_PROP_RO(...)\
    YREFLECTION_AUTO_MEMBER(__VA_ARGS__, YREFLECTION_PROPERTY_GETTER, YREFLECTION_RO_SETTER)

#define YREFLECTION_AUTO_GETSET(...)\
    YREFLECTION_AUTO_MEMBER(__VA_ARGS__, YREFLECTION_GETTER, YREFLECTION_SETTER)

#define YREFLECTION_AUTO_GETSET_(...)\
    YREFLECTION_AUTO_MEMBER(__VA_ARGS__, YREFLECTION_GETTER_, YREFLECTION_SETTER_)

//=================================================================================
// OLD-FASHONED MACROS
//=================================================================================

#define YREFLECTION_MEMBER(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_MEMBER_GETTER(NAME), YREFLECTION_MEMBER_SETTER(NAME))

#define YREFLECTION_MEMBER_RENAMED(TYPE, REFLECTION_NAME, FIELD_NAME)\
    (REFLECTION_NAME, TYPE, TYPE, YREFLECTION_MEMBER_GETTER(FIELD_NAME), YREFLECTION_MEMBER_SETTER(FIELD_NAME))

#define YREFLECTION_ROMEMBER(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_MEMBER_GETTER(NAME), yamail::data::reflection::ignore(obj,val))

#define YREFLECTION_ROMEMBER_RENAMED(TYPE, REFLECTION_NAME, FIELD_NAME)\
    (REFLECTION_NAME, TYPE, TYPE, YREFLECTION_MEMBER_GETTER(FIELD_NAME), yamail::data::reflection::ignore(obj,val))

#define YREFLECTION_PROPERTY(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_PROPERTY_GETTER(NAME), YREFLECTION_PROPERTY_SETTER(NAME))

#define YREFLECTION_ROPROPERTY(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_PROPERTY_GETTER(NAME), yamail::data::reflection::ignore(obj,val))

#define YREFLECTION_GETSET(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_GETTER(NAME), YREFLECTION_SETTER(NAME))

#define YREFLECTION_GET(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_GETTER(NAME), yamail::data::reflection::ignore(obj,val))

#define YREFLECTION_GETSET_(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_GETTER_(NAME), YREFLECTION_SETTER_(NAME))

#define YREFLECTION_GET_(TYPE, NAME)\
    (NAME, TYPE, TYPE, YREFLECTION_GETTER_(NAME), yamail::data::reflection::ignore(obj,val))

#define YREFLECTION_ENUM_MEMBER_GETTER(TYPE, NAME)\
    static_cast<std::add_const_t<std::underlying_type_t<TYPE>>>(yamail::data::reflection::adtUnwrap(obj).NAME)

#define YREFLECTION_ENUM_MEMBER_SETTER(TYPE, NAME)\
    yamail::data::reflection::adtUnwrap(obj).NAME = static_cast<const TYPE>(val)

#define YREFLECTION_ENUM_MEMBER(TYPE, NAME)\
    (NAME, std::underlying_type_t<TYPE>, std::underlying_type_t<TYPE>,\
    (YREFLECTION_ENUM_MEMBER_GETTER(TYPE, NAME)), (YREFLECTION_ENUM_MEMBER_SETTER(TYPE, NAME)))

#define YREFLECTION_WO_MEMBER_RENAMED(TYPE, REFLECTION_NAME, FIELD_NAME) \
    (REFLECTION_NAME, TYPE, TYPE, ((void) obj, TYPE()), yamail::data::reflection::adtUnwrap(obj).FIELD_NAME = val)

#define YREFLECTION_WO_MEMBER(TYPE, FIELD_NAME) \
    YREFLECTION_WO_MEMBER_RENAMED(TYPE, FIELD_NAME, FIELD_NAME)

#define YREFLECTION_WO_MEMBER_RENAMED_WITH_CAST(REFLECTION_TYPE, REFLECTION_NAME, FIELD_TYPE, FIELD_NAME) \
    (REFLECTION_NAME, REFLECTION_TYPE, REFLECTION_TYPE, ((void) obj, REFLECTION_TYPE()), yamail::data::reflection::adtUnwrap(obj).FIELD_NAME = FIELD_TYPE(val))


#define YR_INT_ENUM_ITEM(r, data, elem) elem,

#define YR_INT_ENUM_ITEM_PRETTY(r, data, i, elem) BOOST_PP_COMMA_IF(i) data::elem

#define YR_INT_ADD_ENUM_TO_MAP(r, data, elem) map.insert({elem, BOOST_PP_STRINGIZE(elem)})

#define YR_INT_CASE(r, data, elem) case data::elem: return BOOST_PP_STRINGIZE(elem);

#define YR_INT_CASE_PRETTY(r, data, elem) case data::elem: return BOOST_PP_STRINGIZE(data::elem);

#define YR_INT_ADAPT_ENUM_TO_STRING_STMT(name, sequence)\
        switch (v) {\
        BOOST_PP_SEQ_FOR_EACH(YR_INT_CASE, name, sequence)\
        }\
        return {};

#define YR_INT_ADAPT_ENUM_INLINE_IMPL(name, sequence)\
    inline std::string_view to_string (const name& v) noexcept {\
        YR_INT_ADAPT_ENUM_TO_STRING_STMT(name, sequence)\
    }\
    inline constexpr yamail::data::reflection::enum_traits_helper<name,\
        BOOST_PP_SEQ_FOR_EACH_I(YR_INT_ENUM_ITEM_PRETTY, name, sequence)\
    >  get_enum_reflection_traits(const name&) noexcept;


#define YR_INT_ADAPT_ENUM_IMPL(name, sequence)\
    template <> struct to_string_impl<name> {\
        static constexpr std::string_view apply(const name& v) noexcept {\
            YR_INT_ADAPT_ENUM_TO_STRING_STMT(name, sequence)\
        }\
    };\
    template <> struct enum_traits<name> : enum_traits_helper<name,\
        BOOST_PP_SEQ_FOR_EACH_I(YR_INT_ENUM_ITEM_PRETTY, name, sequence)\
    > {};

#define YREFLECTION_DEFINE_ENUM_INLINE(name, ...)\
    enum class name {\
        BOOST_PP_SEQ_FOR_EACH(YR_INT_ENUM_ITEM, _, BOOST_PP_VARIADIC_TO_SEQ(__VA_ARGS__))\
    };\
    YR_INT_ADAPT_ENUM_INLINE_IMPL(name,  BOOST_PP_VARIADIC_TO_SEQ(__VA_ARGS__))


#define YREFLECTION_ADAPT_ENUM(Enum, ...)\
    namespace yamail::data::reflection {\
    YR_INT_ADAPT_ENUM_IMPL(Enum, BOOST_PP_VARIADIC_TO_SEQ(__VA_ARGS__))\
    }
