#pragma once

#include <boost/type_traits.hpp>
#include <boost/hana/map.hpp>
#include <boost/fusion/adapted.hpp>
#include <boost/fusion/sequence.hpp>
#include <boost/fusion/support/is_sequence.hpp>
#include <boost/fusion/include/is_sequence.hpp>
#include <string>
#include <string_view>
#include <map>
#include <unordered_map>
#include <optional>
#include <sstream>
#include <set>
#include <unordered_set>

namespace yamail::data::reflection {

template <typename T, typename = std::void_t<>>
struct is_std_begin_end_compatible_range : std::false_type {};

template <typename T>
struct is_std_begin_end_compatible_range<T, std::void_t<
    decltype(std::begin(std::declval<T&>())),
    decltype(std::end(std::declval<T&>()))
>> : std::true_type {};

template <typename T>
constexpr auto is_std_begin_end_compatible_range_v = is_std_begin_end_compatible_range<T>::value;

template <typename T, typename = std::void_t<>>
struct is_begin_end_compatible_range : std::false_type {};

template <typename T>
struct is_begin_end_compatible_range<T, std::void_t<
    decltype(begin(std::declval<T&>())),
    decltype(end(std::declval<T&>()))
>> : std::true_type {};

template <typename T>
constexpr auto is_begin_end_compatible_range_v = is_begin_end_compatible_range<T>::value;

template <typename T>
struct is_range : std::bool_constant<
    is_begin_end_compatible_range_v<T> || is_std_begin_end_compatible_range_v<T>
> {};

template <typename T>
constexpr auto is_range_v = is_range<T>::value;

using std::negation;

template <typename T>
using is_fusion_sequence = boost::fusion::traits::is_sequence<T>;

template <typename T>
constexpr auto is_fusion_sequence_v = is_fusion_sequence<T>::value;

template <typename T, typename = std::void_t<>>
struct has_struct_tag : std::false_type {};

template <typename T>
struct has_struct_tag<T, std::enable_if_t<
    std::is_same_v<
        typename boost::fusion::traits::tag_of<T>::type,
        boost::fusion::struct_tag
    >
>> : std::true_type {};

template <typename T>
struct is_fusion_struct : has_struct_tag<T> {};

template <typename T>
constexpr auto is_fusion_struct_v = is_fusion_struct<T>::value;

/**
 * @brief C++20 remove_cvref implementation
 */
template< class T >
struct remove_cvref {
    typedef std::remove_cv_t<std::remove_reference_t<T>> type;
};

template< class T >
using remove_cvref_t = typename remove_cvref<T>::type;


template< class T >
struct is_pair : std::false_type { };

template< class T1 , class T2 >
struct is_pair<std::pair<T1, T2>> : public std::true_type { };

template< class T >
constexpr auto is_pair_v = is_pair<T>::value;

template <typename T>
using is_hana_map = std::is_same<boost::hana::map_tag, boost::hana::tag_of_t<T>>;

template<typename T>
constexpr auto is_hana_map_v = is_hana_map<T>::value;

template <typename T, typename = std::void_t<>>
struct is_map : std::false_type {};

template <typename ...Ts>
struct is_map<std::map<Ts...>> : std::true_type {};

template <typename ...Ts>
struct is_map<std::unordered_map<Ts...>> : std::true_type {};

template <typename ...Ts>
struct is_map<std::multimap<Ts...>> : std::false_type {};

template <typename ...Ts>
struct is_map<std::unordered_multimap<Ts...>> : std::false_type {};

template <typename T>
struct is_map<T, std::void_t<
    typename T::key_type,
    typename T::mapped_type
>> {
    static_assert(std::is_same_v<T, void>,
        "type T looks like Map, but can not identified as Map,"
        "please specify yamail::data::reflection::is_map<> for this type as "
        "std::true_type in case of unique key like in std::map and "
        "std::false_type in case of not unique key like in std::multimap.");
};

template <typename T>
constexpr auto is_map_v = is_map<T>::value;


template <typename T>
struct is_set : std::false_type {};

template <typename ...Ts>
struct is_set<std::set<Ts...>> : std::true_type {};

template <typename ...Ts>
struct is_set<std::unordered_set<Ts...>> : std::true_type {};

template <typename ...Ts>
struct is_set<std::multiset<Ts...>> : std::true_type {};

template <typename ...Ts>
struct is_set<std::unordered_multiset<Ts...>> : std::true_type {};

template <typename T>
constexpr auto is_set_v = is_set<T>::value;


template <typename T>
struct is_string : std::false_type {};

template <typename ...Ts>
struct is_string<std::basic_string<Ts...>> : std::true_type {};

template <typename ...Ts>
struct is_string<std::basic_string_view<Ts...>> : std::true_type {};

template <typename T>
constexpr auto is_string_v = is_string<T>::value;

template <typename T>
struct forward_ref_impl { using type = T; };

template <typename T>
struct forward_ref_impl<T&&> {using type = T;};

template <typename T>
struct forward_ref_impl<T&> { using type = std::reference_wrapper<const T>; };

template <typename T>
struct forward_ref_impl<const T&> { using type = std::reference_wrapper<const T>;};

template <typename T>
using forward_ref_type = typename forward_ref_impl<T>::type;

template <typename T>
constexpr forward_ref_type<T> forward_ref(T&& v) { return v;}

static_assert(std::is_same_v<decltype(forward_ref(std::declval<int>())), int>);
static_assert(std::is_same_v<decltype(forward_ref(std::declval<int&>())), std::reference_wrapper<const int>>);
static_assert(std::is_same_v<decltype(forward_ref(std::declval<const int&>())), std::reference_wrapper<const int>>);
static_assert(std::is_same_v<decltype(forward_ref(std::declval<int&&>())), int>);


struct empty_enum_traits {};

template <typename Enum>
constexpr empty_enum_traits get_enum_reflection_traits(const Enum&) noexcept;

template <typename Enum>
struct enum_traits : decltype(get_enum_reflection_traits(std::declval<const Enum&>())) {};



template <typename Enum, typename = std::void_t<>>
struct is_enum_adapted : std::false_type {};

template <typename Enum>
struct is_enum_adapted<Enum, std::void_t<typename enum_traits<Enum>::type>> : std::true_type {};

/**
 * @brief Indicates if the enumeration has been adapted via the yreflection's enum_traits
 *
 * @tparam Enum --- enumeration type to exam
 */
template <typename Enum>
constexpr auto is_enum_adapted_v = is_enum_adapted<Enum>::value;


/**
 * @brief std::integral_constant-like type for elements of an enumeration
 *
 * @tparam Enum --- enumeration type
 * @tparam items --- enumeration items to hold in
 */
template <typename Enum, Enum ...items>
struct enum_items_constant {
    using value_type = std::initializer_list<Enum>;
    static constexpr value_type value = {items...};
    using type = enum_items_constant;
    constexpr operator const value_type& () const noexcept { return value; }
    constexpr const value_type& operator ()() const noexcept { return value; }
};


/**
 * @brief std::integral_constant-like type for string representation to value mapping
 *
 * @tparam Items --- enum_items_constant specialization type
 */
template <typename Items>
struct enum_items_map_constant {
    using value_type = std::unordered_map<std::string_view, typename Items::value_type::value_type>;
    static const value_type value;
    using type = enum_items_map_constant;
    constexpr operator const value_type& () const noexcept { return value; }
    constexpr const value_type& operator ()() const noexcept { return value; }
};

template <typename Enum, typename = std::enable_if_t<is_enum_adapted_v<Enum>>>
inline decltype(auto) to_string(const Enum& v) noexcept;

template <typename Items>
const typename enum_items_map_constant<Items>::value_type enum_items_map_constant<Items>::value = [] {
            enum_items_map_constant<Items>::value_type retval;
            for (auto v : Items::value) {
                retval.insert({to_string(v), v});
            }
            return retval;
        }();


/**
 * @brief default implementation of a string to enumeration value conversion
 *
 * @tparam Map --- enum_items_map_constant type to get mapping from
 */
template <typename Map>
struct from_string_impl {
    using result_type = std::optional<typename Map::value_type::mapped_type>;
    inline static result_type apply(std::string_view s) noexcept {
        const auto i = Map::value.find(s);
        return i ==  Map::value.end() ? std::nullopt : result_type{i->second};
    }
};


/**
 * @brief default implementation of enumeration value to a string conversion
 *
 * @tparam Enum --- enumeration type
 */
template <typename Enum>
struct to_string_impl {
    static constexpr decltype(auto) apply(const Enum& v) noexcept {
        return to_string(v);
    }
};

/**
 * @brief Forward declaration of enumeration trait type.
 *
 * Trait should be defined like this
 *@code
template <>
struct enum_traits<Enum> {
    using type = enum_traits<Enum>; // using injected-class-name
    using items = enum_items_constant<Enum, Items...>; // should contain items constant for the enumeration
    using from_string = <from string impl>; // should be a functional class with from string coversion implementation
    using to_string = <to string impl>; // should be a functional class with to string coversion implementation
};
 * @endcode
 * @tparam Enum
 */
template <typename Enum>
struct enum_traits;


/**
 * @brief Helper class which helps to implement enum_traits
 *
 * @tparam Enum --- enumeration
 * @tparam Items --- items of the enumeration
 */
template <typename Enum, Enum ...Items>
struct enum_traits_helper {
    using type = enum_traits<Enum>;
    using items = enum_items_constant<Enum, Items...>;
    using from_string = from_string_impl<enum_items_map_constant<items>>;
    using to_string = to_string_impl<Enum>;
};


/**
 * @brief Shortcut to get all items of an enumeration from enum_traits
 *
 * @tparam Enum --- enumeration type to get items
 */
template <typename Enum>
constexpr auto enum_items = enum_traits<Enum>::items::value;


/**
 * @brief Converts enumeration object from string representation
 *
 * Convertion using enum_traits<Enum>::from_string::apply() functional class method
 *
 * @param s --- std::string_view with source string representation
 * @param out --- reference on enumeration value there the result will be placed if succeeded
 * @return true --- value has been coverted successfully
 * @return false --- source string representation has been matched with no enumeration item
 */
template <typename Enum, typename = std::enable_if_t<is_enum_adapted_v<Enum>>>
inline bool from_string(std::string_view s, Enum& out) noexcept {
    static_assert(is_enum_adapted_v<Enum>,
        "enum should be defined via YREFLECTION_DEFINE_ENUM_INLINE"
        " or adapted via YREFLECTION_ADAPT_ENUM"
    );
    if(auto retval = enum_traits<Enum>::from_string::apply(s)) {
        out = *retval;
        return true;
    }
    return false;
}

/**
 * @brief Converts enumeration object from string representation
 *
 * Convertion using enum_traits<Enum>::from_string::apply() functional class method
 *
 * @tparam Enum --- enumeration type to convert to
 * @param s --- std::string_view with source string representation
 * @return Enum --- converted value
 * @throw std::invalid_argument if source string representation has been
 *        matched with no enumeration item
 */
template <typename Enum, typename = std::enable_if_t<is_enum_adapted_v<Enum>>>
inline Enum from_string(std::string_view s) {
    static_assert(is_enum_adapted_v<Enum>,
        "enum should be defined via YREFLECTION_DEFINE_ENUM_INLINE"
        " or adapted via YREFLECTION_ADAPT_ENUM"
    );
    Enum out;
    if (!from_string(s, out)) {
        std::ostringstream err;
        err << "invalid string value for enum \"" << s << '"';
        throw std::invalid_argument(err.str());
    }
    return out;
}

/**
 * @brief Converts enumeration object to string representation
 *
 * Convertion using enum_traits<Enum>::to_string::apply() functional class method
 *
 * @param v --- object to convert
 * @return string with representation
 */
template <typename Enum, typename>
inline decltype(auto) to_string(const Enum& v) noexcept {
    static_assert(is_enum_adapted_v<Enum>,
        "enum should be defined via YREFLECTION_DEFINE_ENUM_INLINE"
        " or adapted via YREFLECTION_ADAPT_ENUM"
    );
    return enum_traits<Enum>::to_string::apply(v);
}

} // namespace yamail::data::reflection
