#pragma once

#include <string>
#include <chrono>
#include <yamail/data/reflection/apply_visitor.h>
#include <boost/lexical_cast.hpp>
#include <boost/hana/map.hpp>
#include <boost/regex.hpp>
#include <boost/range/algorithm/find.hpp>

namespace yamail::data::reflection {

template <typename T>
struct is_duration : std::false_type {};

template <typename R, typename P>
struct is_duration<std::chrono::duration<R, P> > : std::true_type {};

template <typename T>
constexpr auto is_duration_v = is_duration<T>::value;


namespace detail {

template <typename Duration>
struct units {
    static constexpr auto get() {
        constexpr auto u = boost::hana::find(unitMap, boost::hana::type_c<Duration>).value_or(BOOST_HANA_STRING(""));
        return boost::hana::to<const char*>(u);
    }

private:
    static constexpr auto unitMap = boost::hana::make_map(
        boost::hana::make_pair(boost::hana::type_c<std::chrono::nanoseconds>,   BOOST_HANA_STRING("ns")),
        boost::hana::make_pair(boost::hana::type_c<std::chrono::microseconds>,  BOOST_HANA_STRING("us")),
        boost::hana::make_pair(boost::hana::type_c<std::chrono::milliseconds>,  BOOST_HANA_STRING("ms")),
        boost::hana::make_pair(boost::hana::type_c<std::chrono::seconds>,       BOOST_HANA_STRING("s")),
        boost::hana::make_pair(boost::hana::type_c<std::chrono::minutes>,       BOOST_HANA_STRING("m")),
        boost::hana::make_pair(boost::hana::type_c<std::chrono::hours>,         BOOST_HANA_STRING("h"))
    );
};

template <typename Duration>
std::enable_if_t<is_duration_v<Duration>, std::string>
        duration_count(const Duration& value) {
    return std::to_string(value.count());
}

template <typename Duration>
std::string to_string_impl(const Duration& value) {
    return duration_count(value) + units<Duration>::get();
}

template <typename Duration>
Duration from_string_impl(const std::string& value) {
    boost::smatch parsed;
    if (!boost::regex_search(value, parsed, boost::regex(R"(^(?<count>\d+)(?<units>[a-z]*)$)"))) {
        throw std::runtime_error("bad format: " + value);
    }
    const std::string& parsedCount = parsed["count"];
    const std::string& parsedUnits = parsed["units"];

    int count;
    if (!boost::conversion::try_lexical_convert(parsedCount, count)) {
        throw std::runtime_error("bad number: " + parsedCount + " in " + value);
    }

    if (!parsedUnits.empty()) {
#define RETURN_IF_UNITS(__type) \
        if (parsedUnits == units<std::chrono::__type>::get()) { \
            return std::chrono::duration_cast<Duration>(std::chrono::__type(count)); \
        }
        RETURN_IF_UNITS(nanoseconds)
        RETURN_IF_UNITS(microseconds)
        RETURN_IF_UNITS(milliseconds)
        RETURN_IF_UNITS(seconds)
        RETURN_IF_UNITS(minutes)
        RETURN_IF_UNITS(hours)
#undef RETURN_IF_UNITS
        throw std::runtime_error("bad units: " + parsed["units"] + " in " + value);
    }
    return Duration(count);
}

} // namespace detail

template <typename Duration>
std::enable_if_t<is_duration_v<Duration>, std::string>
        to_string(const Duration& value) {
    return detail::to_string_impl(value);
}

template <typename Duration>
std::enable_if_t<is_duration_v<Duration>, Duration>
        from_string(const std::string& value) {
    return detail::from_string_impl<Duration>(value);
}

} // namespace yamail::data::reflection
