#pragma once

#include <yamail/data/reflection/apply_visitor.h>
#include <yamail/data/reflection/details/type_traits.h>
#include <yamail/data/reflection/details/types/reference_wrapper.h>
#include <yamail/data/reflection/apply_visitor.h>
#include <yamail/data/reflection/tag.h>

#include <boost/fusion/include/for_each.hpp>
#include <boost/fusion/container/generation/make_vector.hpp>
#include <boost/fusion/include/at.hpp>
#include <boost/fusion/adapted.hpp>

namespace yamail::data::reflection {

namespace fusion = boost::fusion;

template <typename Adapted, std::size_t I>
struct struct_member_proxy {
    Adapted& obj;
    constexpr decltype(auto) get() const { return fusion::at_c<I>(obj); }
};

template <typename Adapted, std::size_t I>
struct is_reference_wrapper<struct_member_proxy<Adapted, I>> : std::true_type { };

template <typename Adapted, std::size_t I>
using struct_member_name = fusion::extension::struct_member_name<remove_cvref_t<Adapted>, I>;

template <typename Adapted, std::size_t I>
constexpr auto struct_member_tag(const struct_member_proxy<Adapted, I>&) {
    return NamedItemTag<struct_member_name<Adapted, I>>{};
}

template <typename Adapted, std::size_t I>
constexpr auto name(const struct_member_proxy<Adapted, I>&) {
    return struct_member_name<Adapted, I>::call();
}

using fusion::extension::adt_attribute_proxy;

template <typename Adapted, int I, bool Op>
constexpr auto struct_member_tag(const adt_attribute_proxy<Adapted, I, Op>&) {
    return NamedItemTag<struct_member_name<Adapted, I>>{};
}

template <typename Adapted, int I, bool Op>
constexpr auto name(const adt_attribute_proxy<Adapted, I, Op>&) {
    return struct_member_name<Adapted, I>::call();
}

namespace detail {

template <typename T, typename = std::void_t<>>
struct is_adt_adapted : std::false_type {};

template <typename T>
struct is_adt_adapted<T, std::void_t<
    decltype(std::declval<adt_attribute_proxy<std::decay_t<T>, 0, false>>().get()),
    decltype(std::declval<adt_attribute_proxy<std::decay_t<T>, 0, true>>().get())
>>: std::true_type {};

template <typename T>
constexpr auto is_adt_adapted_v = is_adt_adapted<T>::value;

template <typename Adapted, std::size_t... I>
constexpr auto members_impl(Adapted& obj, std::index_sequence<I...>,
        std::enable_if_t<!is_adt_adapted_v<Adapted>>* = nullptr) {
    return fusion::make_vector(struct_member_proxy<Adapted, I>{obj}...);
}

template <typename Adapted, std::size_t... I>
constexpr auto members_impl(Adapted& obj, std::index_sequence<I...>,
        std::enable_if_t<is_adt_adapted_v<Adapted>>* = nullptr) {
    return fusion::make_vector(fusion::at_c<I>(obj)...);
}

} // namespace detail

template <typename T, int N>
struct is_reference_wrapper<adt_attribute_proxy<T, N, true>> : std::true_type { };

/**
 * @brief `apply_visitor_impl` specialization for adt_attribute_proxy setter
 *
 * Since setter operation consumes objects it is needed to provide a buffer object
 * to deserialize to. This specialization implements such behaviour.
 */
template <typename T, typename Visitor, std::size_t N>
struct apply_visitor_impl<adt_attribute_proxy<T, N, false>, Visitor> {
    using proxy = adt_attribute_proxy<T, N, false>;

    template <typename V>
    constexpr static auto get(V&& v) { return v;}

    template <typename V>
    constexpr static auto get(std::reference_wrapper<V> v) { return v.get();}

    template <typename Tag>
    constexpr static void apply(Visitor& v, proxy p, Tag&& tag) {
        auto buf = get(p.get());
        yamail::data::reflection::apply_visitor(v, buf, std::forward<Tag>(tag));
        p = buf;
    }
};

template <typename T, typename Visitor, std::size_t N>
struct apply_visitor_impl<const adt_attribute_proxy<T, N, false>, Visitor> :
    apply_visitor_impl<adt_attribute_proxy<T, N, false>, Visitor> {};

template <typename Adapted>
constexpr decltype(auto) members(Adapted&& obj) {
    using Indices = std::make_index_sequence<decltype(fusion::size(obj))::value>;
    return detail::members_impl(std::forward<Adapted>(obj), Indices{});
}

/**
 * @brief Applies default Visitor to a struct or class adapted by Boost.Fusion.
 *
 * This strategy expands class/structure and applies visitor for each of it's member_proxy.
 */
template <typename T>
struct apply_visitor_default_impl<T, std::enable_if_t<is_fusion_struct_v<T>>> {
    template <typename Obj, typename Visitor, typename Tag>
    static void apply (Visitor&& v, Obj&& value, Tag&& tag) {
        static_assert(is_fusion_struct_v<remove_cvref_t<Obj>>);
        decltype(auto) item_visitor = v.onStructStart(value, tag);
        boost::fusion::for_each(members(value), [&](auto& proxy) {
            yamail::data::reflection::apply_visitor(item_visitor, proxy, struct_member_tag(proxy));
        });
        v.onStructEnd(value, tag);
    }
};

} // namespace yamail::data::reflection
