#pragma once

#include <yamail/data/reflection/apply_visitor.h>
#include <yamail/data/reflection/details/type_traits.h>

#include <boost/variant.hpp>

#if __cplusplus >= 201703L
#include <variant>
#endif

namespace yamail::data::reflection {

template <class T>
struct is_variant : std::false_type { };

template <class ...Ts>
struct is_variant<boost::variant<Ts...>> : std::true_type { };

template <typename Visitor, typename ...Ts>
inline decltype(auto) visit(Visitor&& v, boost::variant<Ts...>& in) {
    return boost::apply_visitor(std::forward<Visitor>(v), in);
}

template <typename Visitor, typename ...Ts>
inline decltype(auto) visit(Visitor&& v, const boost::variant<Ts...>& in) {
    return boost::apply_visitor(std::forward<Visitor>(v), in);
}

#if __cplusplus >= 201703L

template <class ...Ts>
struct is_variant<std::variant<Ts...>> : public std::true_type {};

using std::visit;

#endif

template <typename T>
constexpr auto is_variant_v = is_variant<T>::value;

/**
 * @brief Applies default Visitor for `boost::variant` or `std::variant` objects.
 * Applicable only for a constant objects.
 */
template <typename T>
struct apply_visitor_default_impl<const T, std::enable_if_t<is_variant_v<T>>> {
    template <typename Visitor, typename Tag>
    static void apply (Visitor&& visitor, const T& variant, Tag&& tag) {
        visit([&](auto& v) {
            yamail::data::reflection::apply_visitor(std::forward<Visitor>(visitor), v, std::forward<Tag>(tag));
        }, variant);
    };
};

template <typename T>
struct apply_visitor_default_impl<T, std::enable_if_t<is_variant_v<T>>> {
    template <typename Visitor, typename Tag>
    static void apply (Visitor&& , T& , Tag&& ) {
        static_assert(std::is_void_v<T>, "variant deserialization operation is not supported");
    }
};

} // namespace yamail::data::reflection
