#pragma once

#include <boost/lexical_cast.hpp>
#include <boost/hana/string.hpp>

namespace yamail {
namespace data {
namespace reflection {

struct SequenceItemTag {};

template <typename Name>
struct NamedItemTag {
    using name = Name;
};

template <typename Name>
struct TagValue;

template <typename T>
struct NamedItemTag<TagValue<T>> {
    const T& name;
};

template <typename Name>
inline NamedItemTag<TagValue<Name>> namedItemTag(const Name& name) {
    return NamedItemTag<TagValue<Name>>{name};
}

template <char ...Vs>
constexpr decltype(auto) name(const NamedItemTag<TagValue<boost::hana::string<Vs...>>>& tag) {
    return boost::hana::to<const char*>(tag.name);
}

template <typename ... Args>
constexpr decltype(auto) name(const NamedItemTag<Args...>& ) {
    return NamedItemTag<Args...>::name::call();
}

template <typename T>
inline const T& name(const NamedItemTag<TagValue<T>>& tag){
    return tag.name;
}

template <typename ... Args>
inline auto stringName(const NamedItemTag<Args...>& tag) {
    return name(tag);
}

template <typename T>
inline auto stringName(const NamedItemTag<TagValue<T>>& tag)
        -> std::enable_if_t<
            !std::is_same<
                std::string,
                decltype(boost::lexical_cast<std::string>(tag.name))
            >::value,
            const T&
        > {
    return tag.name;
}

template <typename T>
inline auto stringName(const NamedItemTag<TagValue<T>>& tag)
        -> std::enable_if_t<
            std::is_same<
                std::string,
                decltype(boost::lexical_cast<std::string>(tag.name))
            >::value,
            std::string
        > {
    std::string result;
    if (!boost::conversion::try_lexical_convert<std::string>(tag.name, result)) {
        throw std::runtime_error("Failed to convert tag name to std::string");
    }
    return result;
}

} // namespace reflection
} // namespace data
} // namespace yamail
