#pragma once

#include <yamail/data/reflection.h>
#include <yamail/data/reflection/visitor.h>
#include <yamail/data/reflection/details/urlencoded_traits.h>

#include <boost/lexical_cast.hpp>


namespace yamail::data::serialization {

namespace urlencoded {
using namespace yamail::data::reflection;

struct WriterException: public std::runtime_error {
    using std::runtime_error::runtime_error;
};

inline std::string boolToYesNo(bool b) {
    return b ? "yes" : "no";
}

using BoolToString = std::function<std::string(bool)>;

struct DefaultOptions {
    using DumpValue = std::function<void(const std::string&, const std::string&)>;

    const DumpValue& dumpValue;
    const BoolToString& bts;
    bool throwOnEmptyPointer;

    template<class Value>
    void write(const std::string& name, const Value& val) const {
        if constexpr (std::is_same_v<Value, bool>) {
            dumpValue(name, bts(val));
        } else {
            dumpValue(name, boost::lexical_cast<std::string>(val));
        }
    }
};

template<class Options>
struct Writer: public Visitor {
    const Options& options;

    Writer(const Options& o)
        : options(o)
    { }

    template<typename Value, typename ... Args>
    void onValue(const Value& v, NamedItemTag<Args...> tag) const {
        if constexpr (std::is_pointer_v<Value>) {
            if (v) {
                options.template write<decltype(*v)>(name(tag), *v);
            } else if (options.throwOnEmptyPointer) {
                throw WriterException(std::string("empty pointer with name: ") + name(tag));
            }
        } else {
            options.template write<Value>(name(tag), v);
        }
    }

    template <typename Sequence, typename Named>
    Visitor onSequenceStart(const Sequence& c, Named tag) const {
        static_assert(!unwrap_v<Sequence>, "Sequence type must not be wrapped");

        for(const auto& element : c) {
            onValue(element, tag);
        }

        return Visitor();
    }

    template<typename Map, typename Tag>
    Visitor onMapStart(Map&&, Tag) = delete;

    template<typename Struct, typename ... Args>
    Writer onStructStart(Struct&, NamedItemTag<Args...>) const { return *this; }

    template<template<class> class Optional, class Value, class Named>
    bool onOptional(const Optional<Value>& p, Named) const {
        if constexpr (boost::fusion::traits::is_sequence<unwrap_t<Value>>::value) {
            static_assert(!unwrap_v<Value>, "Optional value type must not be wrapped");

            if (p) {
                Writer writer = *this;
                applyVisitor(*p, writer, namedItemTag("root"));
            }

            return false;
        } else {
            return static_cast<bool>(p);
        }
    }
};

}

template<class T, class S>
void toUrlencodedWithOptions(T& params, S&& opts) {
    urlencoded::Writer writer(std::forward<S>(opts));
    reflection::applyVisitor(params, writer, reflection::namedItemTag("root"));
}

template<class T>
void toUrlencoded(T& params, const urlencoded::DefaultOptions::DumpValue& dumpValue) {
    urlencoded::DefaultOptions opts {
        .dumpValue=dumpValue,
        .bts=urlencoded::boolToYesNo,
        .throwOnEmptyPointer=false
    };

    urlencoded::Writer writer(std::move(opts));
    reflection::applyVisitor(params, writer, reflection::namedItemTag("root"));
}

}
