#pragma once

#include <yamail/data/reflection/reflection.h>
#include <yamail/data/reflection/options.h>
#include <contrib/libs/yajl/api/yajl_gen.h>

#define YREFLECTION_YAJL_CHECK_STATUS(value) \
    yamail::data::serialization::yajl::checkStatus(value, __PRETTY_FUNCTION__, __FILE__, __LINE__)

namespace yamail::data::serialization {

using namespace yamail::data::reflection;

class JsonError : public std::runtime_error {
public:
    JsonError(const std::string& msg) : std::runtime_error(msg) {}
};

namespace yajl {

using Handle = boost::shared_ptr<yajl_gen_t>;

inline Handle createGenerator() {
    Handle result( yajl_gen_alloc(nullptr),  yajl_gen_free );
    if (result.get() == nullptr) {
        throw JsonError("yajl_gen_alloc failed");
    }
    return result;
}

inline const char* statusName(yajl_gen_status value) noexcept {
#define YREFLECTION_YAJL_STATUS_NAME(value) case value: return #value;
    switch (value) {
        YREFLECTION_YAJL_STATUS_NAME(yajl_gen_status_ok)
        YREFLECTION_YAJL_STATUS_NAME(yajl_gen_keys_must_be_strings)
        YREFLECTION_YAJL_STATUS_NAME(yajl_max_depth_exceeded)
        YREFLECTION_YAJL_STATUS_NAME(yajl_gen_in_error_state)
        YREFLECTION_YAJL_STATUS_NAME(yajl_gen_generation_complete)
        YREFLECTION_YAJL_STATUS_NAME(yajl_gen_invalid_number)
        YREFLECTION_YAJL_STATUS_NAME(yajl_gen_no_buf)
        YREFLECTION_YAJL_STATUS_NAME(yajl_gen_invalid_string)
    };
#undef YREFLECTION_YAJL_STATUS_NAME
    return "unknown";
}

inline void checkStatus(int value, const char* function, const char* file, int line) {
   if ( value != yajl_gen_status_ok ) {
       std::ostringstream s;
       s << "yajl error: " << statusName(yajl_gen_status(value)) << " in " << function << " at " << file << ":" << line;
       throw JsonError(s.str());
   }
}

class Buffer {
    Handle h;
    const unsigned char *buf = nullptr;
    std::size_t len = 0;
public:
    using const_iterator = const char*;
    using iterator = const_iterator;
    Buffer(Handle hh) : h(hh) {
        YREFLECTION_YAJL_CHECK_STATUS(yajl_gen_get_buf(h.get(), &buf, &len));
    }
    const_iterator begin() const noexcept { return reinterpret_cast<const_iterator>(buf);}
    const_iterator end() const noexcept { return begin() + len; }
    std::size_t size() const noexcept { return len; }
    std::string str() const { return std::string{begin(), end()}; }
    operator const char* () const noexcept { return begin(); }
    operator std::string () const { return str(); }
    bool operator !() const noexcept { return buf == nullptr; }
};

struct OptForceNull : std::true_type {};

constexpr auto optForceNull = OptForceNull{};

template <typename Options>
class Writer : public Visitor {
public:
    explicit Writer (Handle gen, Options options)
     : options(std::move(options)), gen(gen) {
    }

    template<typename T, typename Tag>
    void apply(const T& value, Tag rootName) {
        YREFLECTION_YAJL_CHECK_STATUS ( yajl_gen_map_open(gen.get()) );
        applyVisitor(value, *this, rootName);
        YREFLECTION_YAJL_CHECK_STATUS ( yajl_gen_map_close(gen.get()) );
    }

    template<typename T>
    void apply(const T & value) {
        applyVisitor(value, *this, SequenceItemTag());
    }

    template<typename Optional, typename Tag>
    bool onOptional(const Optional& p, Tag&& tag) {
        return onNullable(p, std::forward<Tag>(tag));
    }

    template<typename Ptr, typename Tag>
    bool onSmartPointer(const Ptr& p, Tag&& tag) {
        return onNullable(p, std::forward<Tag>(tag));
    }

    template <typename T, typename ... Args>
    void onValue(const T& value, NamedItemTag<Args...> tag) {
        addString(name(tag));
        onValue(value, SequenceItemTag{});
    }

    void onValue(float value, SequenceItemTag tag) {
        onValue(static_cast<double>(value), tag);
    }

    void onValue(double value, SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS(yajl_gen_double(gen.get(), value));
    }

    void onValue(short value, SequenceItemTag tag) {
        onValue(static_cast<long long>(value), tag);
    }

    void onValue(int value, SequenceItemTag tag) {
        onValue(static_cast<long long>(value), tag);
    }

    void onValue(long value, SequenceItemTag tag) {
        onValue(static_cast<long long>(value), tag);
    }

    void onValue(long long l, SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_integer(gen.get(), l) );
    }

    void onValue(unsigned short value, SequenceItemTag tag) {
        onValue(static_cast<unsigned long long>(value), tag);
    }

    void onValue(unsigned int value, SequenceItemTag tag) {
        onValue(static_cast<unsigned long long>(value), tag);
    }

    void onValue(unsigned long value, SequenceItemTag tag) {
        onValue(static_cast<unsigned long long>(value), tag);
    }

    void onValue(unsigned long long value, SequenceItemTag) {
        const auto str = std::to_string(value);
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_number(gen.get(), str.c_str(), str.size()) );
    }

    void onValue(bool value, SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_bool(gen.get(), value) );
    }

    template<typename T>
    void onValue(const T& v, SequenceItemTag) {
        addString(v);
    }

    template<typename Struct, typename ... Args>
    Writer onStructStart(const Struct& p, NamedItemTag<Args...> tag) {
        addString( name(tag) );
        return onStructStart(p, SequenceItemTag{});
    }

    template<typename Struct>
    Writer onStructStart(const Struct&, SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS ( yajl_gen_map_open(gen.get()) );
        return *this;
    }

    template<typename Struct, typename Tag>
    void onStructEnd(const Struct&, Tag) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_map_close(gen.get()) );
    }

    template<typename Map, typename Tag>
    Writer onMapStart(const Map& m, Tag tag) {
        return onStructStart(m, tag);
    }

    template<typename Map, typename Tag>
    void onMapEnd(const Map&, Tag) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_map_close(gen.get()) );
    }

    template<typename Seq, typename ... Args>
    Writer onSequenceStart(const Seq& seq, NamedItemTag<Args...> tag) {
        addString( name(tag) );
        return onSequenceStart(seq, SequenceItemTag());
    }

    template<typename Seq>
    Writer onSequenceStart(const Seq&, SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS(yajl_gen_array_open(gen.get()));
        return *this;
    }

    template<typename Seq, typename Tag>
    void onSequenceEnd(const Seq& , Tag) {
        YREFLECTION_YAJL_CHECK_STATUS(yajl_gen_array_close(gen.get()));
    }

    template <typename Ptree, typename Tag >
    void onPtree(const Ptree& tree, Tag tag) {
        if (tree.size() == 0) {
            onValue(tree.data(), tag);
        } else if (tree.front().first.empty()) {
            auto v = onSequenceStart(tree, tag);
            for (const auto& i : tree) {
                applyVisitor(i.second, v, SequenceItemTag());
            }
            onSequenceEnd(tree, tag);
        } else {
            auto v = onMapStart(tree, tag);
            for (const auto& i : tree) {
                applyVisitor(i.second, v, namedItemTag(i.first));
            }
            onMapEnd(tree, tag);
        }
    }

private:
    void addString(char ch) {
        addString(std::string_view(std::addressof(ch), 1));
    }
    void addString(const char* str) {
        addString(std::string_view(str));
    }
    template <class T>
    std::enable_if_t<!is_string<T>::value> addString (const T& v) {
        addString(std::to_string(v));
    }
    template <class T>
    std::enable_if_t<is_string<T>::value> addString (const T& str) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_string(gen.get(),
                    reinterpret_cast<const unsigned char*>(str.data()),
                    str.size()) );
    }

    template<typename Nullable, typename Tag>
    bool onNullable(const Nullable& p, Tag&& tag) {
        if constexpr (!hasOption<OptForceNull, Options>()) {
            return bool(p);
        }

        if (bool(p)) {
            return true;
        }
        addNull(std::forward<Tag>(tag));
        return false;
    }

    template <typename ... Args>
    void addNull(NamedItemTag<Args...> tag) {
        addString(name(tag));
        addNull(SequenceItemTag{});
    }

    void addNull(SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS(yajl_gen_null(gen.get()));
    }

    Options options;
    Handle gen;
};

template <typename Stream>
inline void print(void* ctx, const char* str, std::size_t len) {
    reinterpret_cast<Stream*>(ctx)->write(str, static_cast<std::streamsize>(len));
}

} // namespace yajl

template <typename T, typename ...Ts>
inline yajl::Buffer toJson(const T& v, std::tuple<Ts...> options = std::tuple<>{}) {
    auto h = yajl::createGenerator();
    yajl::Writer<std::tuple<Ts...>>(h, std::move(options)).apply(v);
    return yajl::Buffer(h);
}

template <typename T, typename ...Ts>
inline yajl::Buffer toJson(const T& v, const std::string& rootName, std::tuple<Ts...> options = std::tuple<>{}) {
    auto h = yajl::createGenerator();
    yajl::Writer<std::tuple<Ts...>>(h, std::move(options)).apply(v, namedItemTag(rootName));
    return yajl::Buffer(h);
}

template <typename Stream, typename T, typename ...Ts>
inline Stream& writeJson(Stream& stream, const T& v, std::tuple<Ts...> options = std::tuple<>{}) {
    using S = typename std::decay<Stream>::type;
    auto h = yajl::createGenerator();
    if (!yajl_gen_config(h.get(), yajl_gen_print_callback, &yajl::print<S>, &stream)) {
        std::ostringstream s;
        s << "yajl error: yajl_gen_config failed in " << __PRETTY_FUNCTION__ << " at " __FILE__ << ":" << __LINE__;
        throw JsonError(s.str());
    }
    yajl::Writer<std::tuple<Ts...>>(h, std::move(options)).apply(v);
    return stream;
}

template <typename Stream, typename T, typename ...Ts>
inline Stream& writeJson(Stream& stream, const T& v, const std::string& rootName, std::tuple<Ts...> options = std::tuple<>{}) {
    using S = typename std::decay<Stream>::type;
    auto h = yajl::createGenerator();
    if (!yajl_gen_config(h.get(), yajl_gen_print_callback, &yajl::print<S>, &stream)) {
        std::ostringstream s;
        s << "yajl error: yajl_gen_config failed in " << __PRETTY_FUNCTION__ << " at " __FILE__ << ":" << __LINE__;
        throw JsonError(s.str());
    }
    yajl::Writer<std::tuple<Ts...>>(h, std::move(options)).apply(v, namedItemTag(rootName));
    return stream;
}

template <typename T, typename Tag, typename Options>
struct JsonChunks {
    JsonChunks(Tag tag, Options options)
     : tag(tag), options(std::move(options)) {
    }

    JsonChunks(const JsonChunks& other) = default;
    JsonChunks(JsonChunks&& other) = default;

    yajl::Handle handle_;
    Tag tag;
    Options options;
    bool firstCall = true;

    yajl::Handle handle() {
        if(handle_ == nullptr) {
             handle_ = yajl::createGenerator();
        }
        return handle_;
    }
    template<typename Tg>
    void onStart(Tg tg) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_map_open(handle().get()) );
        YREFLECTION_YAJL_CHECK_STATUS(
            yajl_gen_string(
                handle().get(),
                reinterpret_cast<const unsigned char*>(name(tg).c_str()),
                name(tg).size()
            )
        );
        onStart(SequenceItemTag{});
    }

    void onStart(SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_array_open(handle().get()) );
    }

    template<typename Tg>
    void onEnd(Tg) {
        onEnd(SequenceItemTag{});
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_map_close(handle().get()) );
    }

    void onEnd(SequenceItemTag) {
        YREFLECTION_YAJL_CHECK_STATUS( yajl_gen_array_close(handle().get()) );
    }

    yajl::Buffer operator()(const boost::optional<T>& v) {
        yajl_gen_clear(handle().get());
        auto writer = yajl::Writer<Options>(handle(), options);

        if( firstCall ) {
            onStart(tag);
            firstCall = false;
        }

        if( v ) {
            writer.apply(*v);
        } else {
            onEnd(tag);
        }

        return yajl::Buffer(handle());
    }
};

template <typename T, typename Tag, typename Options = std::tuple<>>
auto toChunkedJson(Tag tag, Options&& options = Options{}) {
    return JsonChunks<T, Tag, std::decay_t<Options>>(tag, std::forward<Options>(options));
}

}

#undef YREFLECTION_YAJL_CHECK_STATUS
