#pragma once

#include <yamail/data/reflection/reflection.h>

#include <yaml-cpp/yaml.h>

namespace yamail {
namespace data {
namespace serialization {

using namespace yamail::data::reflection;

namespace yaml {

struct RootNodeTag {};

class Writer : public Visitor {
public:
    explicit Writer(YAML::Emitter& emitter) noexcept
        : emitter(emitter)
    {}

    template<typename T>
    void apply(const T& value) {
        applyVisitor(value, *this, RootNodeTag());
    }

    template<typename T, typename... Args>
    void onValue(const T& value, NamedItemTag<Args...> tag) {
        emitter << YAML::Key << name(tag) << YAML::Value ;
        onValue(value, SequenceItemTag{});
    }

    template<typename T, typename Tag>
    void onValue(const T& value, Tag) {
        emitter << value;
    }

    template<typename Tag>
    void onValue(const std::string_view& value, Tag) {
        emitter << std::string(value.begin(), value.end());
    }

    template <typename Struct, typename ... Args>
    Writer onStructStart(const Struct&, NamedItemTag<Args...> tag) {
        emitter << YAML::Key << name(tag) << YAML::Value << YAML::BeginMap;
        return *this;
    }

    template <typename Struct, typename Tag>
    Writer onStructStart(const Struct&, Tag) {
        emitter << YAML::BeginMap;
        return *this;
    }

    template <typename Struct, typename Tag>
    void onStructEnd(const Struct&, Tag) {
        emitter << YAML::EndMap;
    }

    template<typename Map, typename... Args>
    Writer onMapStart(const Map&, NamedItemTag<Args...> tag) {
        emitter << YAML::Key << name(tag) << YAML::Value << YAML::BeginMap;
        return *this;
    }

    template<typename Map>
    Writer onMapStart(const Map&, SequenceItemTag) {
        emitter << YAML::BeginMap;
        return *this;
    }

    template<typename Map, typename Tag>
    void onMapEnd(const Map&, Tag) {
        emitter << YAML::EndMap;
    }

    template<typename Sequence, typename... Args>
    Writer onSequenceStart(const Sequence&, NamedItemTag<Args...> tag) {
        emitter << YAML::Key << name(tag) << YAML::Value << YAML::BeginSeq;
        return *this;
    }

    template<typename Sequence, typename Tag>
    Writer onSequenceStart(const Sequence&, Tag) {
        emitter << YAML::BeginSeq;
        return *this;
    }

    template<typename Sequence, typename Tag>
    void onSequenceEnd(const Sequence&, Tag) {
        emitter << YAML::EndSeq;
    }

    const char* result() const {
        return emitter.c_str();
    }

private:
    YAML::Emitter& emitter;
};

} // yaml namespace

template <typename T>
YAML::Emitter& toYaml(YAML::Emitter& emitter, const T& value) {
    yaml::Writer(emitter).apply(value);
    return emitter;
}

template <typename T>
std::string toYaml(const T& value) {
    YAML::Emitter emitter;
    toYaml(emitter, value);
    return std::string(emitter.c_str(), emitter.size());
}

}}}
