#pragma once

#include <tuple>
#include <map>
#include <vector>
#include <list>

#include <boost/mpl/vector.hpp>
#include <boost/type_erasure/builtin.hpp>
#include <boost/type_erasure/operators.hpp>
#include <boost/type_erasure/any_cast.hpp>

namespace yplatform {
namespace tskv {
namespace detail {

/**
 * TSKV attribute - pair of key and value.
 */
template <typename Key, typename Value>
using attribute = std::tuple<Key, Value>;

/**
 * Attribute's key accessor
 */
template <typename Key, typename Value>
inline const Key& key(const attribute<Key, Value>& v)
{ return std::get<0>(v); }

/**
 * Attribute's value accessor
 */
template <typename Key, typename Value>
inline const Value& value(const attribute<Key, Value>& v)
{ return std::get<1>(v); }

template <typename T>
struct is_attribute : public std::false_type {};

template <typename Key, typename Value>
struct is_attribute<attribute<Key, Value>> : public std::true_type {};

template <typename ... Ts>
struct is_attributes : public std::false_type {};

template <typename Key, typename Value, typename T1, typename ... Ts>
struct is_attributes<attribute<Key,Value>, T1, Ts...> : public is_attributes<T1, Ts...> {};

template <typename Key, typename Value>
struct is_attributes<attribute<Key,Value>> : public std::true_type {};

template <typename ...Ts>
struct is_attributes_sequence : public std::false_type {};

template <typename ...Ts>
struct is_attributes_sequence<std::tuple<Ts...>> : public is_attributes<Ts...> {};

template <typename T>
struct is_attributes_container : public std::false_type {};

template <typename Key, typename Value>
struct is_attributes_container<std::map<Key, Value>> : public std::true_type {};

template <typename Key, typename Value>
struct is_attributes_container<std::vector<attribute<Key, Value>>> : public std::true_type {};

template <typename Key, typename Value>
struct is_attributes_container<std::list<attribute<Key, Value>>> : public std::true_type {};

template <typename Key, typename Value>
struct is_attributes_container<std::vector<std::pair<Key, Value>>> : public std::true_type {};

template <typename Key, typename Value>
struct is_attributes_container<std::list<std::pair<Key, Value>>> : public std::true_type {};

/**
 * Attributes map. A storage for the common attributes.
 */
template <
        typename Char,
        typename Traits = std::char_traits<Char>,
        typename Alloc = std::allocator<Char>>
class basic_attributes_map {
    using map_type = std::map<
        std::basic_string<Char, Traits, Alloc>,
        boost::type_erasure::any<
            boost::mpl::vector<
                boost::type_erasure::copy_constructible<>,
                boost::type_erasure::typeid_<>,
                boost::type_erasure::ostreamable<std::basic_ostream<Char, Traits>>,
                boost::type_erasure::relaxed
            >
        >
    >;

    map_type map;
public:
    typename map_type::const_iterator begin() const
    { return map.begin(); }

    typename map_type::const_iterator end() const
    { return map.end(); }

    template <typename K, typename V>
    basic_attributes_map& operator << (const attribute<K,V>& attr)
    {
        this->map[key(attr)] = value(attr);
        return *this;
    }
};

template <typename C, typename T, typename A>
struct is_attributes_container<basic_attributes_map<C, T, A>> : public std::true_type {};

} // namespace detail
} // namespace tskv
} // namespace yplatform
