#pragma once

#include <iostream>
#include <type_traits>
#include <yamail/data/reflection.h>

namespace yplatform {
namespace tskv {
namespace detail {

/**
 * Base class for converters. Used as a type tag for converter.
 */
struct basic_converter {};

/**
 * Determines if type is converter.
 */
template <typename T>
using is_converter = std::is_base_of<basic_converter, T>;

template <typename Ostream>
using string_type = std::basic_string<typename Ostream::char_type, typename Ostream::traits_type>;

template <typename T1, typename T2>
struct make_converters_helper
{
    using type = std::tuple<T1, T2>;
};

template <typename T>
struct make_converters_helper<T, void>
{
    using type = std::tuple<T>;
};

template <typename KeyValue, typename Value = void>
using make_converters_t = typename make_converters_helper<KeyValue, Value>::type;

/**
 * Little magic for converters storage optimization with std::tuple.
 */
template <typename T>
struct get_converter;

template <typename Converter>
struct get_converter<std::tuple<Converter>>
{
    using tuple = std::tuple<Converter>;

    static auto key(const tuple& t) -> decltype(std::get<0>(t))
    { return std::get<0>(t); }

    static auto value(const tuple& t) -> decltype(std::get<0>(t))
    { return std::get<0>(t); }
};

template <typename KeyConverter, typename ValueConverter>
struct get_converter<std::tuple<KeyConverter, ValueConverter>>
{
    using tuple = std::tuple<KeyConverter, ValueConverter>;

    static auto key(const tuple& t) -> decltype(std::get<0>(t))
    { return std::get<0>(t); }

    static auto value(const tuple& t) -> decltype(std::get<1>(t))
    { return std::get<1>(t); }
};

template <typename Ostream, typename Converters, typename T>
inline Ostream& convert_key(Ostream& s, const Converters& c, T&& key)
{ return get_converter<Converters>::key(c)(s, std::forward<T>(key)); }

template <typename Ostream, typename Converters, typename T>
inline Ostream& convert_value(Ostream& s, const Converters& c, T&& value)
{ return get_converter<Converters>::value(c)(s, std::forward<T>(value)); }

template <typename Ostream, typename Converters, typename ...T>
inline Ostream& convert_value(Ostream& s, const Converters& c, const std::chrono::duration<T...>& value)
{ return get_converter<Converters>::value(c)(s, yreflection::to_string(value)); }

} // namespace detail
} // namespace tskv
} // namespace yplatform
