#pragma once

#include <iostream>
#include <yplatform/tskv/detail/stream_wrapper.h>
#include <yplatform/tskv/detail/basic_converter.h>

namespace yplatform {
namespace tskv {
namespace detail {

/**
 * Template for tskv line io manipulators specialization. It accepts
 *      KeyValueConverter - key and value (except ValueConverter is specified)
 *                          converter.
 *      ValueConverter    - value converter or void if no difference between key
 *                          and value conversion is needed.
 *      Finalization      - action which will be made when the tskv line is
 *                          finished. Typically it occurs at stream_wrapper
 *                          destruction or when tskv::endl is used. The default
 *                          action is omit the std::endl at the end of the line.
 * Initially converters and finalization action must be default constructible.
 * In other cases you can use appropriate overloaded () operators to setup them.
 */
template <
    typename KeyValueConverter,
    typename ValueConverter = void,
    typename Finalization = finalize_without_endl
>
struct iomanip
{
    using converters_type = make_converters_t<KeyValueConverter, ValueConverter>;
    converters_type converters;
    Finalization fin;

    /**
     * Constructs stream_wrapper for given stream in function call style.
     */
    template <typename Ostream>
    typename std::enable_if<
        !is_converter<Ostream>::value && !is_finalizator<Ostream>::value,
        stream_wrapper<Ostream, converters_type, Finalization>
    >::type operator () (Ostream& s) const
    { return {s, converters, fin}; }

    /**
     * Constructs stream_wrapper for given stream in IO manipulator style.
     */
    template <typename Ostream>
    friend stream_wrapper<Ostream, converters_type, Finalization> operator << (
            Ostream& s,
            const iomanip& m)
    { return m(s); }

    /**
     * Constructs new iomanip instance with new different key and value converters
     * according to parameters are provided.
     */
    template <typename OtherKeyConverter, typename OtherValueConverter>
    typename std::enable_if<
            is_converter<OtherKeyConverter>::value
            && is_converter<OtherValueConverter>::value,
    iomanip<OtherKeyConverter, OtherValueConverter, Finalization>>::type operator () (
            OtherKeyConverter key_converter,
            OtherValueConverter value_converter) const
    {
        using Other = iomanip<OtherKeyConverter, OtherValueConverter, Finalization>;
        return Other {
            typename Other::converters_type{std::move(key_converter), std::move(value_converter)},
            fin
        };
    }

    /**
     * Constructs new iomanip instance with new same key and value converter
     * according to parameter is provided.
     */
    template <typename OtherConverter>
    typename std::enable_if<
        is_converter<OtherConverter>::value,
    iomanip<OtherConverter>>::type operator () (OtherConverter converter) const
    {
        using Other = iomanip<OtherConverter, void, Finalization>;
        return Other {
            typename Other::converters_type{std::move(converter)},
            fin
        };
    }

    /**
     * Constructs new iomanip instance with new finalization action.
     */
    template <typename OtherFinalization>
    typename std::enable_if<
        is_finalizator<OtherFinalization>::value,
        iomanip<KeyValueConverter, ValueConverter, OtherFinalization>
    >::type operator () (OtherFinalization fin) const
    { return {converters, std::move(fin)}; }
};

} // namespace detail
} // namespace tskv
} // namespace yplatform
